/*
 * Copyright (c) 2003, 2024, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

/*
 * (C) Copyright IBM Corp. 2003 - All Rights Reserved
 *
 * The original version of this source code and documentation is
 * copyrighted and owned by IBM. These materials are provided
 * under terms of a License Agreement between IBM and Sun.
 * This technology is protected by multiple US and International
 * patents. This notice and attribution to IBM may not be removed.
 */

package sun.font;

import java.text.CharacterIterator;

public abstract class CodePointIterator {
    public static final int DONE = -1;

    public abstract void setToStart();
    public abstract void setToLimit();

    public abstract int next();
    public abstract int prev();

    public abstract int charIndex();

    public static CodePointIterator create(char[] text) {
        return new CharArrayCodePointIterator(text);
    }

    public static CodePointIterator create(char[] text, int start, int limit) {
        return new CharArrayCodePointIterator(text, start, limit);
    }

    public static CodePointIterator create(CharacterIterator iter) {
        return new CharacterIteratorCodePointIterator(iter);
    }
}

final class CharArrayCodePointIterator extends CodePointIterator {
    private char[] text;
    private int start;
    private int limit;
    private int index;

    public CharArrayCodePointIterator(char[] text) {
        this.text = text;
        this.limit = text.length;
    }

    public CharArrayCodePointIterator(char[] text, int start, int limit) {
        if (start < 0 || limit < start || limit > text.length) {
            throw new IllegalArgumentException();
        }

        this.text = text;
        this.start = this.index = start;
        this.limit = limit;
    }

    public void setToStart() {
        index = start;
    }

    public void setToLimit() {
        index = limit;
    }

    public int next() {
        if (index < limit) {
            char cp1 = text[index++];
            if (Character.isHighSurrogate(cp1) && index < limit) {
                char cp2 = text[index];
                if (Character.isLowSurrogate(cp2)) {
                    ++index;
                    return Character.toCodePoint(cp1, cp2);
                }
            }
            return cp1;
        }
        return DONE;
    }

    public int prev() {
        if (index > start) {
            char cp2 = text[--index];
            if (Character.isLowSurrogate(cp2) && index > start) {
                char cp1 = text[index - 1];
                if (Character.isHighSurrogate(cp1)) {
                    --index;
                    return Character.toCodePoint(cp1, cp2);
                }
            }
            return cp2;
        }
        return DONE;
    }

    public int charIndex() {
        return index;
    }
}

// note this has different iteration semantics than CharacterIterator
final class CharacterIteratorCodePointIterator extends CodePointIterator {
    private CharacterIterator iter;

    public CharacterIteratorCodePointIterator(CharacterIterator iter) {
        this.iter = iter;
    }

    public void setToStart() {
        iter.setIndex(iter.getBeginIndex());
    }

    public void setToLimit() {
        iter.setIndex(iter.getEndIndex());
    }

    public int next() {
        char cp1 = iter.current();
        if (cp1 != CharacterIterator.DONE) {
            char cp2 = iter.next();
            if (Character.isHighSurrogate(cp1) && cp2 != CharacterIterator.DONE) {
                if (Character.isLowSurrogate(cp2)) {
                    iter.next();
                    return Character.toCodePoint(cp1, cp2);
                }
            }
            return cp1;
        }
        return DONE;
    }

    public int prev() {
        char cp2 = iter.previous();
        if (cp2 != CharacterIterator.DONE) {
            if (Character.isLowSurrogate(cp2)) {
                char cp1 = iter.previous();
                if (Character.isHighSurrogate(cp1)) {
                    return Character.toCodePoint(cp1, cp2);
                }
                iter.next();
            }
            return cp2;
        }
        return DONE;
    }

    public int charIndex() {
        return iter.getIndex();
    }
}
