/*
 * Copyright (c) 2025, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package jdk.jpackage.internal;

import java.nio.file.Path;
import java.util.Objects;
import java.util.Optional;
import jdk.jpackage.internal.model.AppImageSigningConfig;
import jdk.jpackage.internal.model.ConfigException;
import jdk.jpackage.internal.model.LauncherStartupInfo;

final class AppImageSigningConfigBuilder {

    AppImageSigningConfigBuilder(SigningIdentityBuilder signingIdentityBuilder) {
        this.signingIdentityBuilder = Objects.requireNonNull(signingIdentityBuilder);
    }

    AppImageSigningConfigBuilder entitlements(Path v) {
        entitlements = v;
        return this;
    }

    AppImageSigningConfigBuilder entitlementsResourceName(String v) {
        entitlementsResourceName = v;
        return this;
    }

    AppImageSigningConfigBuilder signingIdentifierPrefix(LauncherStartupInfo mainLauncherStartupInfo) {
        final var pkgName = mainLauncherStartupInfo.packageName();
        if (!pkgName.isEmpty()) {
            signingIdentifierPrefix(pkgName + ".");
        } else {
            signingIdentifierPrefix(mainLauncherStartupInfo.simpleClassName() + ".");
        }
        return this;
    }

    AppImageSigningConfigBuilder signingIdentifierPrefix(String v) {
        signingIdentifierPrefix = v;
        return this;
    }

    Optional<AppImageSigningConfig> create() throws ConfigException {
        final var identityCfg = signingIdentityBuilder.create();
        if (identityCfg.isEmpty()) {
            return Optional.empty();
        } else {
            final var validatedEntitlements = validatedEntitlements();
            return identityCfg.map(cfg -> {
                return new AppImageSigningConfig.Stub(cfg.identity(), signingIdentifierPrefix,
                        validatedEntitlements, cfg.keychain().map(Keychain::name),
                        Optional.ofNullable(entitlementsResourceName).orElse("entitlements.plist"));
            });
        }
    }

    private Optional<Path> validatedEntitlements() throws ConfigException {
        return Optional.ofNullable(entitlements);
    }

    private SigningIdentityBuilder signingIdentityBuilder;
    private Path entitlements;
    private String entitlementsResourceName;
    private String signingIdentifierPrefix;
}
