/*
 * Copyright (c) 2022, 2025, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package jdk.jpackage.internal;

import java.nio.file.Path;
import java.util.List;
import jdk.jpackage.internal.model.Launcher;
import jdk.jpackage.internal.model.MacPackage;
import jdk.jpackage.internal.util.PathUtils;

/**
 * Helper to install launchers as services using "launchd".
 */
public final class MacLaunchersAsServices extends UnixLaunchersAsServices {

    MacLaunchersAsServices(BuildEnv env, MacPackage pkg) {
        super(env.appImageDir(), pkg.app(), List.of(), launcher -> {
            return new MacLauncherAsService(env, pkg, launcher);
        });
    }

    public static Path getServicePListFileName(String bundleIdentifier,
            String launcherName) {
        String baseName = launcherName.replaceAll("[\\s]", "_");
        return Path.of(bundleIdentifier + "-" + baseName + ".plist");
    }

    private static class MacLauncherAsService extends UnixLauncherAsService {

        MacLauncherAsService(BuildEnv env, MacPackage pkg, Launcher launcher) {
            super(pkg.app(), launcher, env.createResource("launchd.plist.template").setCategory(I18N
                    .getString("resource.launchd-plist-file")));

            plistFilename = getServicePListFileName(pkg.app().bundleIdentifier(), getName());

            // It is recommended to set value of "label" property in launchd
            // .plist file equal to the name of this .plist file without the suffix.
            String label = PathUtils.replaceSuffix(plistFilename.getFileName(), "").toString();

            getResource()
                    .setPublicName(plistFilename)
                    .addSubstitutionDataEntry("LABEL", label)
                    .addSubstitutionDataEntry("APPLICATION_LAUNCHER",
                            pkg.asInstalledPackageApplicationLayout().orElseThrow().launchersDirectory().resolve(getName()).toString());
        }

        @Override
        Path descriptorFilePath(Path root) {
            return root.resolve("Library/LaunchDaemons").resolve(plistFilename);
        }

        private final Path plistFilename;
    }
}
