# Copyright 2015 Google LLC.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import datetime
import logging
import socket

from flask import Flask, request
from google.cloud import datastore


app = Flask(__name__)


def is_ipv6(addr):
    """Checks if a given address is an IPv6 address."""
    try:
        socket.inet_pton(socket.AF_INET6, addr)
        return True
    except OSError:
        return False


# [START gae_flex_datastore_app]
@app.route("/")
def index():
    ds = datastore.Client()

    user_ip = request.remote_addr

    # Keep only the first two octets of the IP address.
    if is_ipv6(user_ip):
        user_ip = ":".join(user_ip.split(":")[:2])
    else:
        user_ip = ".".join(user_ip.split(".")[:2])

    entity = datastore.Entity(key=ds.key("visit"))
    entity.update(
        {
            "user_ip": user_ip,
            "timestamp": datetime.datetime.now(tz=datetime.timezone.utc),
        }
    )

    ds.put(entity)
    query = ds.query(kind="visit", order=("-timestamp",))

    results = []
    for x in query.fetch(limit=10):
        try:
            results.append("Time: {timestamp} Addr: {user_ip}".format(**x))
        except KeyError:
            print("Error with result format, skipping entry.")

    output = "Last 10 visits:\n{}".format("\n".join(results))

    return output, 200, {"Content-Type": "text/plain; charset=utf-8"}


# [END gae_flex_datastore_app]


@app.errorhandler(500)
def server_error(e):
    logging.exception("An error occurred during a request.")
    return (
        """
    An internal error occurred: <pre>{}</pre>
    See logs for full stacktrace.
    """.format(
            e
        ),
        500,
    )


if __name__ == "__main__":
    # This is used when running locally. Gunicorn is used to run the
    # application on Google App Engine. See entrypoint in app.yaml.
    app.run(host="127.0.0.1", port=8080, debug=True)
