/*
 * Copyright 2013 two forty four a.m. LLC <http://www.twofortyfouram.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * <http://www.apache.org/licenses/LICENSE-2.0>
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

package com.jfsoftware.ledcontrol;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Handler;
import android.os.SystemClock;
import android.util.Log;
import android.widget.Toast;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import eu.chainfire.libsuperuser.Shell;

/**
 * This is the "fire" BroadcastReceiver for a Locale Plug-in setting.
 *
 * @see com.twofortyfouram.locale.Intent#ACTION_FIRE_SETTING
 * @see com.twofortyfouram.locale.Intent#EXTRA_BUNDLE
 */
public final class FireReceiver extends BroadcastReceiver
{

    /**
     * @param context {@inheritDoc}.
     * @param intent the incoming {@link com.twofortyfouram.locale.Intent#ACTION_FIRE_SETTING} Intent. This
     *            should contain the {@link com.twofortyfouram.locale.Intent#EXTRA_BUNDLE} that was saved by
     *            {@link EditActivity} and later broadcast by Locale.
     */
    @Override
    public void onReceive(final Context context, final Intent intent)
    {
        /*
         * Always be strict on input parameters! A malicious third-party app could send a malformed Intent.
         */

        if (!com.twofortyfouram.locale.Intent.ACTION_FIRE_SETTING.equals(intent.getAction()))
        {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                      String.format(Locale.US, "Received unexpected Intent action %s", intent.getAction())); //$NON-NLS-1$
            }
            return;
        }

        BundleScrubber.scrub(intent);

        final Bundle bundle = intent.getBundleExtra(com.twofortyfouram.locale.Intent.EXTRA_BUNDLE);
        BundleScrubber.scrub(bundle);

        if (PluginBundleManager.isBundleValid(bundle))
        {
            final int red = bundle.getInt(PluginBundleManager.BUNDLE_EXTRA_INT_RED);
            final int green = bundle.getInt(PluginBundleManager.BUNDLE_EXTRA_INT_GREEN);
            final int blue = bundle.getInt(PluginBundleManager.BUNDLE_EXTRA_INT_BLUE);
            if(isOrderedBroadcast()) {
                setLight(red, green, blue, true);
                final int delay = bundle.getInt(PluginBundleManager.BUNDLE_EXTRA_INT_DELAY);
                final Boolean off = bundle.getBoolean(PluginBundleManager.BUNDLE_EXTRA_BOOL_OFF);
                if(delay > 0) {
                    SystemClock.sleep(delay);
                    if(off) {
                        setLight(0, 0, 0, true);
                    }
                }
                setResultCode(TaskerPlugin.Setting.RESULT_CODE_OK);
            } else {
                setLight(red, green, blue, true);
            }
        }
    }

    public static void setLight(final int red, final int green, final int blue, final Boolean sameThread) {
        if(sameThread) {
            if(Shell.SU.available()) {
                Shell.SU.run("echo " + red + " > /sys/class/leds/red/brightness & echo " + green + " > /sys/class/leds/green/brightness & echo " + blue + " > /sys/class/leds/blue/brightness");
            }
        } else {
            AsyncTask.execute(new Runnable() {
                @Override
                public void run() {
                    setLight(red, green, blue, true);
                }
            });
        }
    }
}