// <copyright file="adocommand.cpp" company="Mancier Connections">
// Copyright (c) 2009 All Right Reserved
//
// THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY 
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
// PARTICULAR PURPOSE.
//
// </copyright>
// <author>Patrick Mancier</author>
// <email>teknerd001@gmail.com</email>
// <date>2009-03-21</date>
// <summary>CADOCommand class to set up and execute stored procedures on SQL server via ADO</summary>
#include "ADOCommand.h"

////////////////////////////////////////////////////////////////////////////////////////////////////////
CADOCommand::CADOCommand(void)
{
	Initialize();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
CADOCommand::~CADOCommand(void)
{
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Initialize the CADOCommand class, instantiate the Command object interface
void CADOCommand::Initialize()
{
	CREATE_INSTANCE(m_pCommand,Command);
	m_szCommand = _T("");
	m_iRecordsAffected = 0;
	m_lOptions = adCmdStoredProc;
	m_pRecords = NULL;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Make the options to Execute a stored procedure
void CADOCommand::SetOptionToProcedure()
{
	m_lOptions |= adCmdStoredProc;
	m_lOptions &= ~adCmdText;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Make the options to Execute command text
void CADOCommand::SetOptionToCmdText()
{
	m_lOptions |= adCmdText;
	m_lOptions &= ~adCmdStoredProc;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Process a returned recordset on Execute
void CADOCommand::ClearExecuteNoRecords()
{
	m_lOptions &= ~adExecuteNoRecords;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//   No returned records on Execute
void CADOCommand::SetExecuteNoRecords()
{
	m_lOptions |= adExecuteNoRecords;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set the database connection for the ADO Command inteface to an active ADO Connection 
BOOL CADOCommand::SetDBConnection(CADOConnection *pDB)
{
	try
	{
		m_pCommand->ActiveConnection = pDB->GetConnection();
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set the command text for Execute 
BOOL CADOCommand::SetCommand(CString& szCommand)
{
	m_szCommand = szCommand;
	try
	{
		m_pCommand->CommandText = _bstr_t(szCommand);
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Add a parameter to the Command object for Execute
BOOL CADOCommand::AddParameter(CADOParameter* pParam)
{
	try
	{
		m_pCommand->Parameters->Append(pParam->GetParameter());
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}	

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Cancel any asychronous command to ADO operations (currently not supported)
BOOL CADOCommand::Cancel()
{
	HRESULT hr;
	try
	{
		hr = m_pCommand->Cancel();
		if(hr!=0)
		{
			LogCrackHR(hr);
			return FALSE;
		}
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}

	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Reset the command object
BOOL CADOCommand::Reset()
{
	m_szCommand = _T("");
	m_iRecordsAffected = 0;
	m_lOptions = 0;
	if(m_pRecords)
	{
		m_pRecords->Release();
		m_pRecords = NULL;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Execute the ADO operation
BOOL CADOCommand::Execute(CString szCommandString, CString szFilter)
{
	try
	{
		_variant_t vtRecordsAffected;
		if(!szCommandString.IsEmpty())
		{
			SetCommand(szCommandString);
		}

		//	If we are running with no returned recordset, do not set the ADO Recordset object
		if(m_lOptions & adExecuteNoRecords)
		{
			m_pCommand->Execute(&vtRecordsAffected,NULL,m_lOptions);
			m_pRecords = NULL;
		}
		//	We need the returned recordset, so set the ADO Recordset object upon running Execute
		else
		{
			m_pRecords = m_pCommand->Execute(&vtRecordsAffected,NULL,m_lOptions);
			m_pRecords->Filter = _bstr_t(szFilter);
		}

		m_iRecordsAffected = vtRecordsAffected.iVal;
	}
	catch(_com_error& e)
	{
		/*
		Only check for the exception if there are NO errors 
		Apparently this certain exception messages are normal behavoir for certain providers

		http://support.microsoft.com/default.aspx?scid=KB;EN-US;197459
		when using the SQL Server ODBC driver, the following messages may appear:
		[Microsoft][ODBC SQL Server Driver][SQL Server]Changed database context to 'Pubs'. 
		[Microsoft][ODBC SQL Server Driver][SQL Server]Changed language setting to 'us_english'. 
		*/
		long lErrors;
		BOOL bNoErrors = TRUE;
		lErrors = m_pCommand->ActiveConnection->Errors->Count;
		if(lErrors)
		{
			for(long lErrCount = 0; lErrCount < lErrors; lErrCount++)
			{
				if(m_pCommand->ActiveConnection->Errors->Item[lErrCount]->NativeError != (long)5701)
				{
					m_iRecordsAffected = 0;
					ReportException(e);
					bNoErrors = FALSE;
				}
			}

			return bNoErrors;
		}
	}

	return TRUE;

}
