#include "ADOConnection.h"
// <copyright file="adoconnection.cpp" company="Mancier Connections">
// Copyright (c) 2009 All Right Reserved
//
// THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY 
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
// PARTICULAR PURPOSE.
//
// </copyright>
// <author>Patrick Mancier</author>
// <email>teknerd001@gmail.com</email>
// <date>2009-03-21</date>
// <summary>CADOConnection class to connect to SQL server via ADO</summary>

//	If running SQL Express locally, this should theoretically work
//	For remote servers you will have to change the SERVER name
//	For other databases, change the Initial Catalog
//
//	DRIVER=SQL Native Client;SERVER=.\SQLExpress;Trusted_Connection=Yes;Initial Catalog=ADOTest;

////////////////////////////////////////////////////////////////////////////////////////////////////////
CADOConnection::CADOConnection(void)
{
	Initialize();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
CADOConnection::CADOConnection(CString szConnectionString)
{
	Initialize();
	SetConnectionString(szConnectionString);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
CADOConnection::~CADOConnection(void)
{
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
void CADOConnection::Initialize()
{
	m_bOpen = FALSE;
	m_lNestingLevel = 0;
	m_lRecordsAffected=0;
	m_szConnectionString = _T("");

	CREATE_INSTANCE(m_pConnection,Connection);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL CADOConnection::SetConnectionString(CString szConnectionString)
{
	m_szConnectionString = szConnectionString;
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set the timeout to connect to the SQL server
BOOL CADOConnection::SetConnectionTimeout(long lTimeout)
{
	try
	{
		m_pConnection->CommandTimeout = lTimeout;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}

	return TRUE;
}
////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Open the connection to the SQL server
BOOL CADOConnection::Open(CString szConnectionString, CString szUser, CString szPassword)
{
	SetUser(szUser);
	SetPassword(szPassword);
	SetConnectionString(szConnectionString);
	return Open();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Main Open call to connect to SQL server
BOOL CADOConnection::Open()
{
	HRESULT hr;
	
	try
	{
		//	Attemp to open using options previously set in caller
		hr = m_pConnection->Open(_bstr_t(m_szConnectionString),_bstr_t(m_szUser),_bstr_t(m_szPassword),NULL);
		if(hr!=0)
		{
			LogCrackHR(hr);
			m_bOpen = FALSE;
			return FALSE;
		}
		else
		{
			m_bOpen = TRUE;
		}
	}
	catch(_com_error& e)
	{
		ReportException(e);
		LogAdoErrorImport(m_pConnection);
		return FALSE;
	}

	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Close the SQL server connection
BOOL CADOConnection::Close()
{
	HRESULT hr;
	try
	{
		m_bOpen = FALSE;
		hr = m_pConnection->Close();
		if(hr!=0)
		{
			LogCrackHR(hr);
			return FALSE;
		}
	}
	catch(_com_error& e)
	{
		ReportException(e);
		LogAdoErrorImport(m_pConnection);
		return FALSE;
	}

	return TRUE;
}
////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Use the Cancel method to terminate execution of an asynchronous method call: that is, 
//	a method invoked with the adAsyncConnect, adAsyncExecute, or adAsyncFetch option.
BOOL CADOConnection::Cancel()
{
	HRESULT hr;
	try
	{
		m_bOpen = FALSE;
		hr = m_pConnection->Cancel();
		if(hr!=0)
		{
			LogCrackHR(hr);
			return FALSE;
		}
	}
	catch(_com_error& e)
	{
		ReportException(e);
		LogAdoErrorImport(m_pConnection);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set or remove an attribute that we need on the ADO Connection::Attribute property
BOOL CADOConnection::SetOrRemoveAttribute(BOOL bMode, long lAttribute)
{
	try
	{
		if(bMode)
		{
			m_pConnection->Attributes |= lAttribute;
		}
		else
		{
			m_pConnection->Attributes &= ~lAttribute;
		}
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}

	return TRUE;
}
////////////////////////////////////////////////////////////////////////////////////////////////////////
//  adXactAbortRetaining
//  Performs retaining aborts by calling RollbackTrans to automatically start a new transaction. Not all providers support this behavior.
BOOL CADOConnection::AbortRetaining(BOOL bMode)
{
	return SetOrRemoveAttribute(bMode,adXactAbortRetaining);
}
////////////////////////////////////////////////////////////////////////////////////////////////////////
// adXactCommitRetaining
// Performs retaining commits by calling CommitTrans to automatically start a new transaction. Not all providers support this behavior.
BOOL CADOConnection::CommitRetaining(BOOL bMode)
{
	return SetOrRemoveAttribute(bMode,adXactCommitRetaining);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Begin a database transaction (NOT TESTED)
BOOL CADOConnection::BeginTransaction()
{
	try
	{
		m_lNestingLevel = m_pConnection->BeginTrans();
	}
	catch(_com_error& e)
	{
		m_lNestingLevel = -1;
		ReportException(e);
		return FALSE;
	}

	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Commit all current changes to the database  (NOT TESTED)
BOOL CADOConnection::Commit()
{
	HRESULT hr;

	try
	{
		hr = m_pConnection->CommitTrans();
		if(hr!=0)
		{
			LogCrackHR(hr);
			return FALSE;
		}
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}

	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Rollback any changes that we have currently made to the database  (NOT TESTED)
BOOL CADOConnection::Rollback()
{
	HRESULT hr;

	try
	{
		hr = m_pConnection->RollbackTrans();
		if(hr!=0)
		{
			LogCrackHR(hr);
			return FALSE;
		}
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}

	return TRUE;
}
////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
Options parameter settings, a combination of CommandTypeEnum and ExecuteOptionEnum values

Taken from http://msdn.microsoft.com/en-us/library/ms675946(VS.85).aspx

------------------------------------------------------------------------------------------------------------
CommandTypeEnum values
------------------------------------------------------------------------------------------------------------
adCmdUnspecified	Does not specify the command type argument.
adCmdText			Evaluates CommandText as a textual definition of a command or stored procedure call.
adCmdTable			Evaluates CommandText as a table name whose columns are all returned by an internally generated SQL query.
adCmdStoredProc		Evaluates CommandText as a stored procedure name.
adCmdUnknown		Default. Indicates that the type of command in the CommandText property is not known.
adCmdFile			Evaluates CommandText as the file name of a persistently stored Recordset. Used with Recordset.Open or Requery only.
adCmdTableDirect	Evaluates CommandText as a table name whose columns are all returned. Used with Recordset.Open or Requery only. 
					To use the Seek method, the Recordset must be opened with adCmdTableDirect.
					This value cannot be combined with the ExecuteOptionEnum value adAsyncExecute.

------------------------------------------------------------------------------------------------------------
ExecuteOptionEnum
------------------------------------------------------------------------------------------------------------
adAsyncExecute			Indicates that the command should execute asynchronously.  This value cannot be combined with the 
						CommandTypeEnum value adCmdTableDirect.
adAsyncFetch			Indicates that the remaining rows after the initial quantity specified in the CacheSize property 
						should be retrieved asynchronously.
adAsyncFetchNonBlocking	Indicates that the main thread never blocks while retrieving. If the requested row has not been retrieved, the current row automatically moves 
						to the end of the file.  If you open a Recordset from a Stream containing a persistently stored Recordset, adAsyncFetchNonBlocking 
						will not have an effect; the operation will be synchronous and blocking.
						adAsynchFetchNonBlocking has no effect when the adCmdTableDirect option is used to open the Recordset.
adExecuteNoRecords		Indicates that the command text is a command or stored procedure that does not return rows (for example, a command that only inserts data). 
						If any rows are retrieved, they are discarded and not returned.
						adExecuteNoRecords can only be passed as an optional parameter to the Command or Connection Execute method.
adExecuteStream			Indicates that the results of a command execution should be returned as a stream.
						adExecuteStream can only be passed as an optional parameter to the Command Execute method.
adExecuteRecord			Indicates that the CommandText is a command or stored procedure that returns a single row which should be returned 
						as a Record object.
adOptionUnspecified		Indicates that the command is unspecified.
*/
//	This method is for doing very simple queries and running stored procedures.  
_RecordsetPtr CADOConnection::Execute(CString szCommandString)
{
	VARIANT vtRecords;
	_RecordsetPtr rsp;

	try
	{
		rsp = m_pConnection->Execute(szCommandString.AllocSysString() ,&vtRecords, (adCmdText));
		m_lRecordsAffected = vtRecords.lVal;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		LogAdoErrorImport(m_pConnection);
		return NULL;
	}

	return rsp;
}
