// <copyright file="adomanager.cpp" company="Mancier Connections">
// Copyright (c) 2009 All Right Reserved
//
// THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY 
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
// PARTICULAR PURPOSE.
//
// </copyright>
// <author>Patrick Mancier</author>
// <email>teknerd001@gmail.com</email>
// <date>2009-03-21</date>
// <summary>CADOManager class, manages all ADO operations, base class for use on inheritable table classes</summary>
// ADOInterface.cpp : main project file.

#include "adomanager.h"
#include "adoparameter.h"

//////////////////////////////////////////////////////////////////////////////////////////////////////////
CADOManager::CADOManager(CADOConnection *pDB)
{
	Initialize();
	m_pConnection = pDB;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
CADOManager::CADOManager()
{
	Initialize();
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
CADOManager::~CADOManager()
{
	if(m_pCommand)
	{
		delete m_pCommand;
	}

	if(m_pRecordset)
	{
		delete m_pRecordset;
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL CADOManager::Initialize()
{
	m_pCommand = new CADOCommand();
	m_pConnection = NULL;	
	m_pRecordset = NULL;	
	m_iQueryType = 0;	
	m_iOperation = 0;	
	m_iLastError = 0;	
	m_szLastError = _T("");	
	m_szQuery = _T("");		
	m_szTable = _T("");		

	return TRUE;
}		

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Derive the procedure name from the table we are using and the type of databae operation we want to do
//	<TableName>_<Database operation:DELETE,INSERT,SELECT,UPDATE>
CString CADOManager::GetProcedureName()
{
	CString szProcedureName;
	CString szOperation;

	switch(m_iOperation)
	{
		case Operation_Delete:
			szOperation = "Delete";
		break;
		case Operation_Insert:
			szOperation = "Insert";
        break;
		case Operation_Select:
			szOperation = "Select";
        break;
		case Operation_Update:
			szOperation = "Update";
        break;
	}

	szProcedureName = GetTable() + "_" + szOperation; 

	return szProcedureName;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Get the values of the fields for the current record in the recordset
void CADOManager::GetRecordValues()
{
	if(m_iOperation == Operation_Select)
	{
		GetSELECTValues();
	}

	if(m_iQueryType == QueryType_Procedure || m_iQueryType == QueryType_SQLQuery )
	{
		GetUserValues();
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Get the next return in the returned recordset
BOOL CADOManager::GetNextRecord()
{
	if(m_pRecordset)
	{
		//	No records were returned, force an exit
		if(m_pRecordset->IsNoRecords())
		{
			return FALSE;
		}

		//	As long as we are not at the end of the recordset, continue getting records
		if(!m_pRecordset->IsEOF())
		{
			GetRecordValues();
			m_pRecordset->MoveNext();
		}
		else
		{
			return FALSE;
		}
	}
	else
	{
		m_szLastError = "ERROR: CADORecordset object undefined";
		return FALSE;
	}

	return TRUE;
}

	
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Get an integer value from the recordset based on the column name
BOOL CADOManager::GetValue(CString szColumnName,int& iValue)
{
	if(m_pRecordset)
	{
		if(!m_pRecordset->GetValue(szColumnName,iValue))
		{
			m_szLastError.Format(_T("ERROR: Could not get field value %s as integer from recordset"),szColumnName);
			return FALSE;
		}
	}
	else
	{
		m_szLastError = "ERROR: CADORecordset class is undefined";
		return FALSE;
	}

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Get a string value from the recordset based on the column name
BOOL CADOManager::GetValue(CString szColumnName,CString& szValue)
{
	if(m_pRecordset)
	{
		if(!m_pRecordset->GetValue(szColumnName,szValue))
		{
			m_szLastError.Format(_T("ERROR: Could not get field value %s as string from recordset"),szColumnName);
			return FALSE;
		}
	}
	else
	{
		m_szLastError = "ERROR: CADORecordset class is undefined";
		return FALSE;
	}

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL CADOManager::GetValue(CString szColumnName,FILETIME& ftValue)
{
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Add an integer parameter to the procedure call
void CADOManager::AddInputParameter(CString szParameterName, int iValue)
{
	CADOParameter *pParam;
	pParam = new CADOParameter(szParameterName,adParamInput,iValue);
	m_pCommand->AddParameter(pParam);
	delete pParam;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Add a string parameter to the procedure call
void CADOManager::AddInputParameter(CString szParameterName, CString szValue)
{
	CADOParameter *pParam;
	pParam = new CADOParameter(szParameterName,adParamInput,szValue);
	m_pCommand->AddParameter(pParam);
	delete pParam;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Add a FILETIME parameter to the procedure call (currently not supported)
void CADOManager::AddInputParameter(CString szParameterName, FILETIME ftValue)
{
	/*
	CADOParameter *pParam;
	pParam = new CADOParameter(szParameterName,adParamInput,ftValue);
	m_pCommand->AddParameter(pParam);
	delete pParam;
	*/
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Run a user defined procedure
BOOL CADOManager::ADO_USERPROC(CString szProc,CString szFilter)
{
	SetUserParameters();
	if(m_pCommand && m_pConnection)
	{
		m_pCommand->Reset();
		m_pCommand->SetOptionToProcedure();
		SetQueryType(QueryType_Procedure);
		m_pCommand->SetDBConnection(m_pConnection);
		if(!m_pCommand->Execute(szProc))
		{
			m_szLastError = m_pCommand->GetLastError();
			m_iLastError = m_pCommand->GetLastHRError();
			return FALSE;
		}
		GetUserValues();
	}
	else
	{
		m_szLastError = "Undefined CADOCommand pointer, aborting";
		return FALSE;
	}

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Run a user defined query
BOOL CADOManager::ADO_USERQUERY(CString szQuery,CString szFilter)
{
	if(m_pCommand && m_pConnection)
	{
		m_pCommand->Reset();
		m_pCommand->SetOptionToCmdText();
		SetQueryType(QueryType_SQLQuery);
		SetUserParameters();
		m_pConnection->GetConnection()->CursorLocation = adUseClient;
		m_pCommand->SetDBConnection(m_pConnection);
		if(!m_pCommand->Execute(szQuery))
		{
			m_szLastError = m_pCommand->GetLastError();
			m_iLastError = m_pCommand->GetLastHRError();

			return FALSE;
		}
		GetUserValues();
	}
	else
	{
		m_szLastError = "Undefined CADOCommand or CADOConnection pointer, aborting";
		return FALSE;
	}

	return TRUE;
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Run a basic DELETE, INSERT, SELECT or UPDATE procedure on a specific table
BOOL CADOManager::RunTableOperation(enumOperations ADOOperation,CString szFilter)
{
   	CString szProcName;

	if(m_pCommand)
	{
		m_pCommand->Reset();
		m_pCommand->SetDBConnection(m_pConnection);
		m_pCommand->SetOptionToProcedure();
		if(ADOOperation!=Operation_Select)
		{
			m_pCommand->SetExecuteNoRecords();
		}
		SetQueryType(QueryType_Table);
		SetOperation(ADOOperation);
		szProcName = GetProcedureName();
		if(!m_pCommand->Execute(szProcName,szFilter))
		{
			m_szLastError = m_pCommand->GetLastError();
			m_iLastError = m_pCommand->GetLastHRError();
			return FALSE;
		}

		if(ADOOperation==Operation_Select)
		{
			if(m_pRecordset)
			{
				delete m_pRecordset;
			}
			m_pRecordset = new CADORecordset(m_pCommand->GetReturnedRecordset());
		}
	}
	else
	{
		m_szLastError = "Undefined CADOCommand pointer, aborting";
		return FALSE;
	}

	return TRUE;
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Run a DELETE operation on a table
BOOL CADOManager::ADO_DELETE()
{
	SetDELETEParameters();
	return RunTableOperation(Operation_Delete);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Run an INSERT operation on a table
BOOL CADOManager::ADO_INSERT()
{
	SetINSERTParameters();
	return RunTableOperation(Operation_Insert);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Run a SELECT operation on a table
BOOL CADOManager::ADO_SELECT(CString szFilter)
{
	BOOL bSuccess;

	SetSELECTParameters();
	bSuccess = RunTableOperation(Operation_Select);
	GetSELECTValues();
	return bSuccess;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Run an UPDATE operation on a table
BOOL CADOManager::ADO_UPDATE()
{
    SetUPDATEParameters();
	return RunTableOperation(Operation_Update);
}

