// <copyright file="adoparameter.cpp" company="Mancier Connections">
// Copyright (c) 2009 All Right Reserved
//
// THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY 
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
// PARTICULAR PURPOSE.
//
// </copyright>
// <author>Patrick Mancier</author>
// <email>teknerd001@gmail.com</email>
// <date>2009-03-21</date>
// <summary>CADOParameter class, wrapper class for ADO Parameter object</summary>
#include "ADOParameter.h"

////////////////////////////////////////////////////////////////////////////////////////////////////////
CADOParameter::CADOParameter(void)
{
	Initialize();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
CADOParameter::~CADOParameter(void)
{
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Create an integer parameter
CADOParameter::CADOParameter(CString szName,ParameterDirectionEnum enumDirection,int iValue)
{
	Initialize();
	SetName(szName);
	SetDirection(enumDirection);
	SetValue(iValue);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Create a string parameter
CADOParameter::CADOParameter(CString szName,ParameterDirectionEnum enumDirection,CString szValue)
{
	Initialize();
	SetName(szName);
	SetDirection(enumDirection);
	SetValue(szValue);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Create a FILTIME parameter (current not supported)
CADOParameter::CADOParameter(CString szName,ParameterDirectionEnum enumDirection,FILETIME ftValue)
{
	Initialize();
	SetName(szName);
	SetDirection(enumDirection);
	SetValue(ftValue);
}
////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Initialize the CADOParameter class, instantiate the ADO Parameter interface
void CADOParameter::Initialize()
{
	CREATE_INSTANCE(m_pParameter,Parameter);
}
////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set the datatype this parameter is
BOOL CADOParameter::SetType(DataTypeEnum enumType)
{
	try
	{
		m_pParameter->Type = enumType;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set the direction that this parameter is for
BOOL CADOParameter::SetDirection(ParameterDirectionEnum enumDirection)
{
	try
	{
		m_pParameter->Direction = enumDirection;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set the value of this parameter
BOOL CADOParameter::SetValue(VARIANT& vtValue)
{
	try
	{
		m_pParameter->Value = vtValue;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set the integer value of this parameter
BOOL CADOParameter::SetValue(int iValue)
{
	_variant_t vtValue;

	SetTypeToInteger();
	vtValue.vt = VT_I2; 
	vtValue.iVal = iValue;
	SetSize(-1);
	try
	{
		m_pParameter->Value = vtValue;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set a boolean parameter (current not supported)
BOOL CADOParameter::SetBoolean(BOOL bValue)
{
	_variant_t vtValue;
	SetTypeToBoolean();
	vtValue.vt = VT_BOOL;
	vtValue.boolVal = bValue;
	return SetValue(vtValue);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set a BSTR parameter (current not supported)
BOOL CADOParameter::SetValue(BSTR& bstrValue)
{
	BOOL bSuccess;
	_variant_t vtValue;

	SetTypeToBSTR();
	vtValue.vt = VT_BSTR;
	vtValue.bstrVal = bstrValue;
	bSuccess = SetValue(vtValue);
	SysFreeString(vtValue.bstrVal);
	return bSuccess;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set a charactor array parameter (current not supported)
BOOL CADOParameter::SetValue(char *cValue,int iSize)
{
	_variant_t vtValue;
	SetTypeToVarChar();
	SetSize(iSize);
	return SetValue(vtValue);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set a FILETIME parameter (current not supported)
BOOL CADOParameter::SetValue(FILETIME ftValue)
{
	_variant_t vtValue;

	return SetValue(vtValue);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set a CString parameter
BOOL CADOParameter::SetValue(CString szValue)
{
	BOOL bSuccess;
	_variant_t vtValue;

	SetTypeToBSTR();

	if(!szValue.IsEmpty())
	{
		vtValue.vt = VT_BSTR;
		SetSize(szValue.GetLength());
	}
	else
	{
		vtValue.vt = VT_NULL;
		SetSize(0);
	}

	vtValue.bstrVal = _bstr_t((char*)(LPCSTR)szValue);
	bSuccess = SetValue(vtValue);

	SysFreeString(vtValue.bstrVal);

	return bSuccess;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set the numeric scale of this parameter
BOOL CADOParameter::SetNumericScale(unsigned char ucNumericScale)
{
	try
	{
		m_pParameter->NumericScale = ucNumericScale;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set the precision of this parameter
BOOL CADOParameter::SetPrecision(unsigned char ucPrecision)
{
	try
	{
		m_pParameter->Precision = ucPrecision;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL CADOParameter::SetSize(int iSize)
{
	try
	{
		m_pParameter->Size = iSize;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL CADOParameter::SetName(CString szName)
{
	try
	{
		m_pParameter->Name = _bstr_t((char*)(LPCSTR)szName);
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

