// <copyright file="adorecordset.cpp" company="Mancier Connections">
// Copyright (c) 2009 All Right Reserved
//
// THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY 
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
// PARTICULAR PURPOSE.
//
// </copyright>
// <author>Patrick Mancier</author>
// <email>teknerd001@gmail.com</email>
// <date>2009-03-21</date>
// <summary>CADORecordset class, wrapper for ADO Recordset object</summary>
#include "ADORecordset.h"
////////////////////////////////////////////////////////////////////////////////////////////////////////
CADORecordset::CADORecordset(_RecordsetPtr& pRecordset)
{
	m_pRecordset = pRecordset;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
CADORecordset::CADORecordset(void)
{

}

////////////////////////////////////////////////////////////////////////////////////////////////////////
CADORecordset::~CADORecordset(void)
{
	Close();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set the way the cursor should be used on recordset (not tested)
BOOL CADORecordset::SetCursorMode(CursorTypeEnum enumCursorType)
{
	try
	{
		m_pRecordset->CursorType = enumCursorType;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;

}

////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL CADORecordset::SetCursorToDynamic()
{
	return SetCursorMode(adOpenDynamic);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL CADORecordset::SetCursorToKeyset()
{
	return SetCursorMode(adOpenKeyset);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL CADORecordset::SetCursorToStatic()
{
	return SetCursorMode(adOpenStatic);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL CADORecordset::SetCursorToForwardOnly()
{
	return SetCursorMode(adOpenForwardOnly);
}
////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL CADORecordset::SetCursorToUnspecified()
{
	return SetCursorMode(adOpenUnspecified);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Move to the first record in the recordset
BOOL CADORecordset::MoveFirst()
{
HRESULT hr;

	try
	{
		hr = m_pRecordset->MoveFirst();
		LogCrackHR(hr);
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Move to the last record in the recordset
BOOL CADORecordset::MoveLast()
{
HRESULT hr;

	try
	{
		hr = m_pRecordset->MoveLast();
		LogCrackHR(hr);
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;

}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Move to the next record in the recordset
BOOL CADORecordset::MoveNext()
{
HRESULT hr;

	try
	{
		if(!IsEOF())
		{
			hr = m_pRecordset->MoveNext();
			LogCrackHR(hr);
		}

		//	If we are at the end of the recordset, we need to fail because next call will fail
		if(IsEOF())
		{
			return FALSE;
		}
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}

	return TRUE;

}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Move to the previous record in the recordset
BOOL CADORecordset::MovePrev()
{
	HRESULT hr;
	try
	{
		if(!IsBOF())
		{
			hr = m_pRecordset->MovePrevious();
			LogCrackHR(hr);
		}

		//	If we are at beginning of recordset, we need to fail because next call will fail
		if(IsBOF())
		{
			return FALSE;
		}
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Move to a specific record index, this be very slow with many records!
BOOL CADORecordset::MoveTo(int iRecordsToMove)
{
	if(MoveFirst())
	{
		for(int iRecordCount = 0; iRecordCount < iRecordsToMove; iRecordCount++)
		{
			if(!IsEOF())
			{
				if(!MoveNext())
				{
					return FALSE;
				}
			}
		}
	}

	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Get an integer value from a field
BOOL CADORecordset::GetValue(LPCSTR pszName, int& iValue)
{
	_variant_t vtValue;

	try
	{
		vtValue = m_pRecordset->Fields->Item[pszName]->GetValue();
		switch(vtValue.vt)
		{
			case VT_BOOL:
				iValue = (int)vtValue.boolVal;
			break;
			case VT_INT:
				iValue = vtValue.intVal;
			break;
			case (VT_INT|VT_BYREF):
				iValue = *vtValue.pintVal;
			break;
			case VT_UINT:
				iValue = (int)vtValue.uintVal;
			break;
			case (VT_UINT|VT_BYREF):
				iValue = (int)*vtValue.puintVal;
			break;
			case VT_UI1:
				iValue = (int)vtValue.bVal;
			break;
			case (VT_UI1|VT_BYREF):
				iValue = (int)*vtValue.pbVal;
			break;
			case VT_I1:
				iValue = (int)vtValue.cVal;
			break;
			case (VT_I1|VT_BYREF):
				iValue = (int)*vtValue.pcVal;
			break;
			case VT_UI2:
				iValue = (int)vtValue.uiVal;
			break;
			case (VT_UI2|VT_BYREF):
				iValue = (int)*vtValue.puiVal;
			break;
			case VT_I2:
				iValue = (int)vtValue.iVal;
			break;
			case (VT_I2|VT_BYREF):
				iValue = (int)*vtValue.piVal;
			break;
			case VT_UI4:
				iValue = (int)vtValue.uintVal;
			break;
			case (VT_UI4|VT_BYREF):
				iValue = (int)*vtValue.puintVal;
			break;
			case VT_I4:
				iValue = (int)vtValue.lVal;
			break;
			case (VT_I4|VT_BYREF):
				iValue = (int)*vtValue.plVal;
			break;
			case VT_I8:
				iValue = (int)vtValue.llVal;
			break;
			case (VT_I8|VT_BYREF):
				iValue = (int)*vtValue.pllVal;
			break;
			case VT_UI8:
				iValue = (int)vtValue.ullVal;
			break;
			case (VT_UI8|VT_BYREF):
				iValue = (int)*vtValue.pullVal;
			break;
		}
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}

	return TRUE;
}
////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Get a CString value from a field
BOOL CADORecordset::GetValue(LPCSTR pszName, CString& szValue)
{
	_variant_t vtValue;
	
	try
	{
		vtValue = m_pRecordset->Fields->Item[pszName]->GetValue();
		switch(vtValue.vt)
		{
			case VT_BSTR:
				szValue = vtValue.bstrVal;
			break;
			case (VT_BSTR | VT_BYREF):
				szValue = *vtValue.pbstrVal;
			break; 
			case VT_NULL:
			case (VT_NULL | VT_BYREF):
			case VT_EMPTY:
			case (VT_EMPTY | VT_BYREF):
				szValue = _T("");
			break;
			case VT_BOOL:
				szValue.Format(_T("%s"),(vtValue.boolVal?"TRUE":"FALSE"));
			break;
			case (VT_BOOL | VT_BYREF):
				szValue.Format(_T("%s"),(*vtValue.pboolVal?"TRUE":"FALSE"));
			break; 
			case VT_INT:
				szValue.Format(_T("%d"),vtValue.intVal);
			break;
			case (VT_INT | VT_BYREF):
				szValue.Format(_T("%d"),*vtValue.pintVal);
			break; 
			case VT_UINT:
				szValue.Format(_T("%d"),vtValue.uintVal);
			break;
			case (VT_UINT | VT_BYREF):
				szValue.Format(_T("%d"),*vtValue.puintVal);
			break; 
			case VT_UI1:
				szValue.Format(_T("%d"),(int)vtValue.bVal);
			break;
			case (VT_UI1 | VT_BYREF):
				szValue.Format(_T("%d"),(int)*vtValue.pbVal);
			break; 
			case VT_I1:
				szValue.Format(_T("%c"),vtValue.cVal);
			break;
			case (VT_I1 | VT_BYREF):
				szValue.Format(_T("%c"),*vtValue.pcVal);
			break; 
			case VT_UI2:
				szValue.Format(_T("%d"),vtValue.uiVal);
			break;
			case (VT_UI2 | VT_BYREF):
				szValue.Format(_T("%d"),*vtValue.puiVal);
			break; 
			case VT_I2:
				szValue.Format(_T("%d"),vtValue.iVal);
			break;
			case (VT_I2 | VT_BYREF):
				szValue.Format(_T("%d"),*vtValue.piVal);
			break; 
			case VT_UI4:
				szValue.Format(_T("%d"),vtValue.uintVal);
			break;
			case (VT_UI4 | VT_BYREF) :
				szValue.Format(_T("%d"),*vtValue.puintVal);
			break;
			case VT_I4:
				szValue.Format(_T("%d"),vtValue.intVal);
			break;
			case (VT_I4 | VT_BYREF):
				szValue.Format(_T("%d"),*vtValue.pintVal);
			break; 
			case VT_UI8:
				szValue.Format(_T("%d"),vtValue.ullVal);
			break;
			case (VT_UI8 | VT_BYREF):
					szValue.Format(_T("%d"),*vtValue.pullVal);
			break; 
			case VT_I8:
				szValue.Format(_T("%d"),vtValue.llVal);
			break;
			case (VT_I8 | VT_BYREF):
				szValue.Format(_T("%d"),*vtValue.pllVal);
			break; 
			break; 
			case VT_R4:
				szValue.Format(_T("%f"),vtValue.fltVal);
			break; 
			case (VT_R4 | VT_BYREF):
				szValue.Format(_T("%d"),*vtValue.pfltVal);
			break; 
			case VT_R8:
				szValue.Format(_T("%f"),vtValue.dblVal);
			break; 
			case (VT_R8 | VT_BYREF):
				szValue.Format(_T("%f"),*vtValue.pdblVal);
			break; 
			default:
				szValue = "Current not supported";
			break;
			/*
			case (VT_CY | VT_BYREF):
			case VT_CY:
			case VT_DECIMAL:
			break; 
			case (VT_DECIMAL | VT_BYREF):
			break; 
			case VT_ERROR:
			break; 
			case (VT_ERROR | VT_BYREF):
			break; 
			case VT_BOOL:
			break; 
			case VT_DATE:
			break; 
			case (VT_DATE | VT_BYREF):
			break; 
			case VT_DISPATCH:
			break; 
			case (VT_DISPATCH | VT_BYREF):
			break; 
			case VT_VARIANT :
			break;
			case (VT_VARIANT | VT_BYREF):
			break;
			case VT_UNKNOWN:
			break; 
			case (VT_UNKNOWN | VT_BYREF):
			break; 
			*/
		}
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}

	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Are we at the beginning of the recordset?
BOOL CADORecordset::IsBOF()
{
	try
	{
		return m_pRecordset->BOF;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Are we at the end of the recordset?
BOOL CADORecordset::IsEOF()
{
	try
	{
		return m_pRecordset->adoEOF;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Are there no records in this recordset?
BOOL CADORecordset::IsNoRecords()
{
	return ((IsBOF()==TRUE)&&(IsEOF()==TRUE));
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Set the filter to apply to this recordset
BOOL CADORecordset::SetFilter(CString szString)
{
	try
	{
		m_pRecordset->Filter = _bstr_t((char*)(LPCSTR)szString);
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;

}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Get the number of records in the recordset
long CADORecordset::GetRecordCount()
{
	try
	{
		return m_pRecordset->RecordCount;
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return -1;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Close the recordset
BOOL CADORecordset::Close()
{
	HRESULT hr;

	try
	{
		hr = m_pRecordset->Close();
		LogCrackHR(hr);
	}
	catch(_com_error& e)
	{
		ReportException(e);
		return FALSE;
	}
	return TRUE;
}
