/*
 **     Copyright (C) 1995  Rick Romero and Carnegie Mellon University
 **
 **     This program is free software; you can redistribute it and/or modify
 **     it under the terms of the GNU General Public License as published by
 **     the Free Software Foundation; either version 1, or (at your option)
 **     any later version.
 **     
 **     This program is distributed in the hope that it will be useful,
 **     but WITHOUT ANY WARRANTY; without even the implied warranty of
 **     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 **     GNU General Public License for more details.
 **     
 **     You should have received a copy of the GNU General Public License
 **     along with this program; if not, write to the Free Software
 **     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <values.h>

#include "dkdtree.h"


#define DKDLEAF_MALLOC_SIZE 100

static DKDLeaf_p DKDLeafFreeList = NULL;

SplayNode_p DKDFindKthRecurse(SplayNode_p, int, int *);
SplayNode_p DKDFindMedian(SplayTree_p, int);
void        DKDRebalanceLowerLevels(DKDNode_p, DKDNode_p, int32, int32);
void        DKDRebuildTree(DKDTree_p);
int         DKDRecurseRebuild(SplayTree_p, SplayTree_p, int, int32,
			      int32, int32);
void        DKDRecurseCopyAndFreeElements(SplayNode_p, SplayTree_p);
void        DKDRecurseCreateAndFree(SplayNode_p, SplayTree_p, int);
void        DKDCreateSortTreeAndFree(DKDTree_p, SplayTree_p);
void        DKDCopyAndFreeTreeElements(SplayTree_p, SplayTree_p);
SplayNode_p DKDFindLastOf(SplayTree_p, SplayNode_p, SplayNode_p, int, int *);
SplayNode_p DKDSkipRepeats(SplayTree_p, SplayNode_p, int, int *);
void        DKDRecurseDestroy(SplayNode_p, int);
int         dkd_compare_nodes(void *obj1, void *obj2);
int         dkd_compare_leaves(void *obj1, void *obj2);
int         dkd_compare_leaves1(void *obj1, void *obj2);
int         dkd_compare_leaves2(void *obj1, void *obj2);
int         dkd_compare_leaves3(void *obj1, void *obj2);


#ifdef DEBUG
int rebalance_counts = 0;
int rebuild_counts = 0;
int inserts = 0;
int deletes = 0;
#endif



int dkd_compare_nodes(void *obj1, void *obj2) {
  
  /* Check to see if either is a point, and not a range */
  if (((DKDNode_p) obj1)->left == ((DKDNode_p) obj1)->right) {
    if (((DKDNode_p) obj2)->left <= ((DKDNode_p) obj1)->left) {
      if (((DKDNode_p) obj2)->right > ((DKDNode_p) obj1)->left) {
	return 0;
      } else {
	return 1;
      }
    } else {
      return -1;
    }
  } else if (((DKDNode_p) obj2)->left == ((DKDNode_p) obj2)->right) {
    if (((DKDNode_p) obj1)->left <= ((DKDNode_p) obj2)->left) {
      if (((DKDNode_p) obj1)->right > ((DKDNode_p) obj2)->left) {
	return 0;
      } else {
	return -1;
      }
    } else {
      return 1;
    }
  } else {
    /* We're comparing intervals, which should all be non-overlapping, so this
     * is a bit easier.
     */
    return ((((DKDNode_p) obj1)->left == ((DKDNode_p) obj2)->left) ? 0 :
	    (((DKDNode_p) obj1)->left > ((DKDNode_p) obj2)->left) ? 1 : -1);
  }
/*  if (((DKDNode_p) obj2)->left <= ((DKDNode_p) obj1)->left) {
    if (((DKDNode_p) obj2)->right > ((DKDNode_p) obj1)->left) {
      return 0;
    } else {
      return 1;
    }
  } else {
    return -1;
  }
*/

}

int dkd_compare_leaves(void *obj1, void *obj2) {
  if (((DKDLeaf_p) obj1)->location[0] ==
      ((DKDLeaf_p) obj2)->location[0]) {
    if (((DKDLeaf_p) obj1)->location[1] ==
	((DKDLeaf_p) obj2)->location[1]) {
      if (((DKDLeaf_p) obj1)->location[2] ==
	  ((DKDLeaf_p) obj2)->location[2]) {
	return ((((DKDLeaf_p) obj1)->location[3] ==
		 ((DKDLeaf_p) obj2)->location[3]) ? 0 :
		((((DKDLeaf_p) obj1)->location[3] <
		  ((DKDLeaf_p) obj2)->location[3]) ? -1 : 1));
      } else return ((((DKDLeaf_p) obj1)->location[2] <
		      ((DKDLeaf_p) obj2)->location[2]) ? -1 : 1);
    } else return ((((DKDLeaf_p) obj1)->location[1] <
		    ((DKDLeaf_p) obj2)->location[1]) ? -1 : 1);
  } else return ((((DKDLeaf_p) obj1)->location[0] <
		  ((DKDLeaf_p) obj2)->location[0]) ? -1 : 1);

}

int dkd_compare_leaves1(void *obj1, void *obj2) {
  if (((DKDLeaf_p) obj1)->location[1] ==
      ((DKDLeaf_p) obj2)->location[1]) {
    if (((DKDLeaf_p) obj1)->location[2] ==
	 ((DKDLeaf_p) obj2)->location[2]) {
      return ((((DKDLeaf_p) obj1)->location[3] ==
	       ((DKDLeaf_p) obj2)->location[3]) ? 0 :
	      ((((DKDLeaf_p) obj1)->location[3] <
		((DKDLeaf_p) obj2)->location[3]) ? -1 : 1));
    } else return ((((DKDLeaf_p) obj1)->location[2] <
		    ((DKDLeaf_p) obj2)->location[2]) ? -1 : 1);
  } else return ((((DKDLeaf_p) obj1)->location[1] <
		  ((DKDLeaf_p) obj2)->location[1]) ? -1 : 1);
}

int dkd_compare_leaves2(void *obj1, void *obj2) {
  if (((DKDLeaf_p) obj1)->location[2] ==
      ((DKDLeaf_p) obj2)->location[2]) {
    return ((((DKDLeaf_p) obj1)->location[3] ==
	     ((DKDLeaf_p) obj2)->location[3]) ? 0 :
	    ((((DKDLeaf_p) obj1)->location[3] <
	      ((DKDLeaf_p) obj2)->location[3]) ? -1 : 1));
  } else return ((((DKDLeaf_p) obj1)->location[2] <
		  ((DKDLeaf_p) obj2)->location[2]) ? -1 : 1);
}

int dkd_compare_leaves3(void *obj1, void *obj2) {
  return ((((DKDLeaf_p) obj1)->location[3] ==
	   ((DKDLeaf_p) obj2)->location[3]) ? 0 :
	  ((((DKDLeaf_p) obj1)->location[3] <
	    ((DKDLeaf_p) obj2)->location[3]) ? -1 : 1));
}


typedef int (*compare_function)(void *, void *);
compare_function dkd_compare_leaves_array[DKD_DIMENSIONS] = \
{  dkd_compare_leaves, dkd_compare_leaves1, dkd_compare_leaves2,
     dkd_compare_leaves3  };




#ifdef DEBUG
int count_points(SplayNode_p node) {
  int c = 0;

  if (node->left)
    c = count_points(node->left);

  if (node->right)
    return (1 + c + count_points(node->right));

  return (1+c);
}
#endif



DKDLeaf_p DKDCreateLeaf() {
  DKDLeaf_p tmp;

  if (DKDLeafFreeList) {
    tmp = DKDLeafFreeList;
    DKDLeafFreeList = (DKDLeaf_p) tmp->obj_list.next;
  } else {
    int i;
    tmp = (DKDLeaf_p) malloc(sizeof(DKDLeaf_t) * DKDLEAF_MALLOC_SIZE);
    tmp += (DKDLEAF_MALLOC_SIZE-1);
    tmp->obj_list.next = (DKDObjList_p) DKDLeafFreeList;
    tmp--;
    for (i=DKDLEAF_MALLOC_SIZE-2; i; i--, tmp--)
      tmp->obj_list.next = (DKDObjList_p) (tmp+1);
    DKDLeafFreeList = tmp+1;
  }
  return tmp;
}


void DKDFreeLeaf(DKDLeaf_p leaf) {
  if (leaf) {
    leaf->obj_list.next = (DKDObjList_p) DKDLeafFreeList;
    DKDLeafFreeList = leaf;
  }
}



void DKDRecurseDestroy(SplayNode_p snode, int dimensions) {
  if (snode==NULL)
    return;

  if (dimensions>0) {
    
    if (snode->left) {
      DKDRecurseDestroy(snode->left, dimensions);
    }

    DKDRecurseDestroy(((DKDNode_p) snode->obj)->stree->root, dimensions - 1);
    free(((DKDNode_p) snode->obj)->stree);
    free(snode->obj);

    if (snode->right) {
      DKDRecurseDestroy(snode->right, dimensions);
    }

    SplayFreeNode(snode);
  } else {

    if (snode->left)
      DKDRecurseDestroy(snode->left, dimensions);
    
    if (snode->right)
      DKDRecurseDestroy(snode->right, dimensions);

    DKDFreeLeaf((DKDLeaf_p) snode->obj);

    SplayFreeNode(snode);

  }
}


void DKDDestroyTree(DKDTree_p dkdtree) {
  
  if (dkdtree) {
    if (dkdtree->stree) {
      if (dkdtree->stree->root)
	DKDRecurseDestroy(dkdtree->stree->root, dkdtree->dimensions-1);
      free(dkdtree->stree);
    }
    free(dkdtree);
  }

}





DKDTree_p CreateDKDTree(int d) {
  DKDTree_p dkdtree;
  DKDNode_p node;
  SplayTree_p stree;
  int i;

  dkdtree = (DKDTree_p) malloc(sizeof(DKDTree_t));
  dkdtree->stree = stree = CreateSplay(dkd_compare_nodes);
  dkdtree->uppernodes = 1;
  dkdtree->lowernodes = LOWER_NODES_MIN;
  dkdtree->points = 0;
  dkdtree->dimensions = d;
  dkdtree->nodes = 1;
  dkdtree->inserts = dkdtree->deletes = 0;
  dkdtree->maxmods = LOWER_NODES_MIN *  8;
  for (i = d-1; i>0; i--) {
    node = (DKDNode_p) malloc(sizeof(DKDNode_t));
    node->dimensions = i;
    if (i > 1) {
      node->nodes = 1;
      node->maxnodes = 2;
    } else {
      node->nodes = 0;
      node->maxnodes = LOWER_NODES_MIN;
    }
    node->points = 0;
    node->maxpoints = 1 << i; /* Max points allowed is 2 per low-level   */
                              /* tree, or 2^d at the top level           */
    node->inserts = 0;
    node->deletes = 0;

    node->left = -MAXINT32;
    node->right = MAXINT32;

    if (i > 1) {
      node->stree = CreateSplay(dkd_compare_nodes);
    } else {
      node->stree = CreateSplay(dkd_compare_leaves);
    }

    SplayInsert(stree, (void *) node);

    stree = node->stree;

  }

  return dkdtree;

}


/*  Find the median element in the splay tree.  Just go until you reach the
 ** (count)'th element.
 */
SplayNode_p DKDFindKthRecurse(SplayNode_p node, int count, int *curnode) {
  SplayNode_p retval;

  if (node->left != NULL) {
    retval = DKDFindKthRecurse(node->left, count, curnode);
    if (*curnode >= count)
      return retval;
  }

  (*curnode)++;
  if (*curnode >= count) {
    return (SplayNode_p) node;
  }

  if (node->right != NULL) {
    return DKDFindKthRecurse(node->right, count, curnode);
  }

  return NULL;
}



/*
 ** Get the median element of the splay tree
 */
SplayNode_p DKDFindMedian(SplayTree_p tree, int points) {
  SplayNode_p node;
  int i;
  i = 0;

  if (tree->root)
    node = DKDFindKthRecurse(tree->root, (points + 1) / 2, &i);
  else
    return NULL;

  return node;
}


#define INLINE 0x1
void DKDRecurseCopyAndFreeElements(SplayNode_p oldnode, SplayTree_p newstree) {
  
  if (oldnode->left) {
#if (INLINE&1)
    if (oldnode->left->left) {
#if (INLINE&2)
      if (oldnode->left->left->left) {
#if (INLINE&4)
	if (oldnode->left->left->left->left) {
	  DKDRecurseCopyAndFreeElements(oldnode->left->left->left->left,
					newstree);
	}
	if (oldnode->left->left->left->right) {
	  DKDRecurseCopyAndFreeElements(oldnode->left->left->left->right,
					newstree);
	}

	SplayInsert(newstree, oldnode->left->left->left->obj);

	SplayFreeNode(oldnode->left->left->left);
#else  /* INLINE&4 */
	DKDRecurseCopyAndFreeElements(oldnode->left->left->left, newstree);
#endif /* INLINE&4 */

      }
      if (oldnode->left->left->right) {
#if (INLINE&4)
	if (oldnode->left->left->right->left) {
	  DKDRecurseCopyAndFreeElements(oldnode->left->left->right->left,
					newstree);
	}
	if (oldnode->left->left->right->right) {
	  DKDRecurseCopyAndFreeElements(oldnode->left->left->right->right,
					newstree);
	}

	SplayInsert(newstree, oldnode->left->left->right->obj);

	SplayFreeNode(oldnode->left->left->right);

#else  /* INLINE&4 */
      DKDRecurseCopyAndFreeElements(oldnode->left->left->right, newstree);
#endif /* INLINE&4 */

      }

      SplayInsert(newstree, oldnode->left->left->obj);
      
      SplayFreeNode(oldnode->left->left);
    
#else  /* INLINE & 2 */
      DKDRecurseCopyAndFreeElements(oldnode->left->left, newstree);
#endif /* INLINE & 2 */
    }

    if (oldnode->left->right) {
#if (INLINE&2)
      if (oldnode->left->right->left) {
#if (INLINE&4)
	if (oldnode->left->right->left->left) {
	  DKDRecurseCopyAndFreeElements(oldnode->left->right->left->left,
					newstree);
	}
	if (oldnode->left->right->left->right) {
	  DKDRecurseCopyAndFreeElements(oldnode->left->right->left->right,
					newstree);
	}

	SplayInsert(newstree, oldnode->left->right->left->obj);

	SplayFreeNode(oldnode->left->right->left);
#else  /* INLINE & 4 */
	DKDRecurseCopyAndFreeElements(oldnode->left->right->left, newstree);
#endif /* INLINE & 4 */

      }

      if (oldnode->left->right->right) {
#if (INLINE&4) 
	if (oldnode->left->right->right->left) {
	  DKDRecurseCopyAndFreeElements(oldnode->left->right->right->left,
					newstree);
	}
	if (oldnode->left->right->right->right) {
	  DKDRecurseCopyAndFreeElements(oldnode->left->right->right->right,
					newstree);
	}

	SplayInsert(newstree, oldnode->left->right->right->obj);

	SplayFreeNode(oldnode->left->right->right);

#else  /* INLINE & 4 */
	DKDRecurseCopyAndFreeElements(oldnode->left->right->right, newstree);
#endif /* INLINE & 4 */

      }

      SplayInsert(newstree, oldnode->left->right->obj);

      SplayFreeNode(oldnode->left->right);
    
#else  /* INLINE & 2 */
      DKDRecurseCopyAndFreeElements(oldnode->left->right, newstree);
#endif /* INLINE & 2 */
    }

    SplayInsert(newstree, oldnode->left->obj);

    SplayFreeNode(oldnode->left);

#else  /* INLINE & 1 */
    DKDRecurseCopyAndFreeElements(oldnode->left, newstree);
#endif /* INLINE & 1 */
  }



  if (oldnode->right) {
#if (INLINE&1)
    if (oldnode->right->left) {
#if (INLINE&2)
      if (oldnode->right->left->left) {
#if (INLINE&4)
	if (oldnode->right->left->left->left) {
	  DKDRecurseCopyAndFreeElements(oldnode->right->left->left->left,
					newstree);
	}
	if (oldnode->right->left->left->right) {
	  DKDRecurseCopyAndFreeElements(oldnode->right->left->left->right,
					newstree);
	}

	SplayInsert(newstree, oldnode->right->left->left->obj);

	SplayFreeNode(oldnode->right->left->left);
#else  /* INLINE & 4 */
	DKDRecurseCopyAndFreeElements(oldnode->right->left->left, newstree);
#endif /* INLINE & 4 */

      }
      if (oldnode->right->left->right) {
#if (INLINE&4)
	if (oldnode->right->left->right->left) {
	  DKDRecurseCopyAndFreeElements(oldnode->right->left->right->left,
					newstree);
	}
	if (oldnode->right->left->right->right) {
	  DKDRecurseCopyAndFreeElements(oldnode->right->left->right->right,
					newstree);
	}

	SplayInsert(newstree, oldnode->right->left->right->obj);

	SplayFreeNode(oldnode->right->left->right);

#else  /* INLINE & 4 */
	DKDRecurseCopyAndFreeElements(oldnode->right->left->right, newstree);
#endif /* INLINE & 4 */
      }

      SplayInsert(newstree, oldnode->right->left->obj);

      SplayFreeNode(oldnode->right->left);
    
#else  /* INLINE & 2 */
      DKDRecurseCopyAndFreeElements(oldnode->right->left, newstree);
#endif /* INLINE & 2 */
    }

    if (oldnode->right->right) {
#if (INLINE&2)
      if (oldnode->right->right->left) {
#if (INLINE&4)
	if (oldnode->right->right->left->left) {
	  DKDRecurseCopyAndFreeElements(oldnode->right->right->left->left,
					newstree);
	}
	if (oldnode->right->right->left->right) {
	  DKDRecurseCopyAndFreeElements(oldnode->right->right->left->right,
					newstree);
	}

	SplayInsert(newstree, oldnode->right->right->left->obj);

	SplayFreeNode(oldnode->right->right->left);
#else  /* INLINE & 4 */
	DKDRecurseCopyAndFreeElements(oldnode->right->right->left, newstree);
#endif /* INLINE & 4 */
      }

      if (oldnode->right->right->right) {
#if (INLINE&4)
	if (oldnode->right->right->right->left) {
	  DKDRecurseCopyAndFreeElements(oldnode->right->right->right->left,
					newstree);
	}
	if (oldnode->right->right->right->right) {
	  DKDRecurseCopyAndFreeElements(oldnode->right->right->right->right,
					newstree);
	}

	SplayInsert(newstree, oldnode->right->right->right->obj);

	SplayFreeNode(oldnode->right->right->right);

#else  /* INLINE & 4 */
	DKDRecurseCopyAndFreeElements(oldnode->right->right->right, newstree);
#endif /* INLINE & 4 */
      }

      SplayInsert(newstree, oldnode->right->right->obj);

      SplayFreeNode(oldnode->right->right);
    
#else  /* INLINE & 2 */
      DKDRecurseCopyAndFreeElements(oldnode->right->right, newstree);
#endif /* INLINE & 2 */
    }

    SplayInsert(newstree, oldnode->right->obj);

    SplayFreeNode(oldnode->right);

#else  /* INLINE & 1 */
    DKDRecurseCopyAndFreeElements(oldnode->right, newstree);
#endif /* INLINE & 1 */
  }

  SplayInsert(newstree, oldnode->obj);

  SplayFreeNode(oldnode);

}



void DKDCopyAndFreeTreeElements(SplayTree_p oldstree, SplayTree_p newstree) {
  if (oldstree) {
    if (oldstree->root) {
      DKDRecurseCopyAndFreeElements(oldstree->root, newstree);
    }
    free(oldstree);
  }
}


SplayNode_p DKDFindLastOf(SplayTree_p stree, SplayNode_p orig,
			  SplayNode_p curnode, int dim, int *count)
{
  SplayNode_p lastnode;

  if (curnode->left) {
    lastnode = DKDFindLastOf(stree, orig, curnode->left, dim, count);
  }

  if (((DKDLeaf_p) curnode->obj)->location[dim] ==
      ((DKDLeaf_p) orig->obj)->location[dim]) {
    (*count)++;
    if (curnode->right) {
      if (((DKDLeaf_p) curnode->right->leftmin->obj)->location[dim] ==
	  ((DKDLeaf_p) orig->obj)->location[dim]) {
	return DKDFindLastOf(stree, orig, curnode->right, dim, count);
      } else {
	return curnode;
      }
    } else {
      return curnode;
    }
  } else {
    return lastnode;
  }

}



SplayNode_p DKDSkipRepeats(SplayTree_p stree, SplayNode_p snode,
			   int dim, int *count) {

  if (snode->right) {
    if (((DKDLeaf_p) snode->right->leftmin->obj)->location[dim] ==
	((DKDLeaf_p) snode->obj)->location[dim]) {
      return DKDFindLastOf(stree, snode, snode->right, dim, count);
    } else {
      return snode;
    }
  }
  return snode;
}


/* Recursively go down and rebuild all the different trees.
 * First split up the current tree, and for each split on the current tree,
 * go down and rebuild its children
 */
int DKDRecurseRebuild(SplayTree_p newtree, SplayTree_p sorttree,
		       int dimensions, int32 uppernodes, int32 lowernodes,
		       int32 totalpoints)
{
  DKDNode_p newnode;
  SplayNode_p splitnode;
  SplayTree_p newstree, tmpstree;
  int i, count, done = 0;
  int32 oldright = -MAXINT32;
  
  if (dimensions > 0) {

#ifdef DEBUG
    fflush(stdout);
    printf("Splitting trees at dimension %d, total points = %d\n",
	   dimensions, totalpoints);
    fflush(stdout);
#endif

    for (i = 0;
	 (i < uppernodes) && (totalpoints - done > 0) && (sorttree->root);
	 i++) {
      /* count = (totalpoints - done) / (uppernodes - i);*/
      count = 0;


#ifdef DEBUG
      fflush(stdout);
      if (count_points(sorttree->root) != totalpoints - done) {
	printf("oops, Points don't add up\n");
      }
      printf("Current split: %d th node, %d nodes done, %d points remaining\n",
	     (totalpoints - done) / (uppernodes - i), i, totalpoints - done);
      fflush(stdout);

#endif

      newnode = (DKDNode_p) malloc(sizeof(DKDNode_t));
      newnode->inserts = newnode->deletes = 0;
      if (dimensions > 1) {
	newnode->maxnodes = 2 * uppernodes;
	newnode->stree = CreateSplay(dkd_compare_nodes);
      } else {
	newnode->maxnodes = 2 * lowernodes;
	newnode->stree = CreateSplay(dkd_compare_leaves);
      }
      newnode->dimensions = dimensions;

      if (totalpoints - done > 1) {
	splitnode =
	  DKDFindKthRecurse(sorttree->root, (totalpoints - done) / 
			    (uppernodes - i), &count);
	/* Could have been that there were duplicates of the same thing.
	 * Need to move them over as well, so that intervals don't overlap
	 * (pain in the ass :-( )
	 */
	SplayNode(sorttree, splitnode);
	count = 0;
	splitnode = DKDSkipRepeats(sorttree, splitnode, dimensions, &count);

	newstree = CreateSplay(dkd_compare_leaves_array[dimensions]);

	/* Bring this guy to the top for splitting */
	SplayNode(sorttree, splitnode);
	/* And split the trees                     */
	newstree->root = sorttree->root->right;
	if (newstree->root != NULL)
	  newstree->root->parent = NULL;
	sorttree->root->right = NULL;

	if ((totalpoints - done) / (uppernodes - i))
	  newnode->points = (totalpoints - done) / (uppernodes - i) + count;
	else
	  newnode->points = 1 + count;


#ifdef DEBUG
	if (newnode->points != (count_points(sorttree->root))) {
	  printf("oops right here\n");
	}
#endif

	newnode->maxpoints = (2 * newnode->points < LOWER_NODES_MIN) ?
	  LOWER_NODES_MIN : 2 * newnode->points;

	/* just set this to the old right */
	newnode->left = oldright;

	if ((i + 1 < uppernodes) && (totalpoints - newnode->points - done > 0))
	  /* add the minimum of the greater tree to the maximum of the lower
	   * tree and divide by two to split between them
	   */
	  newnode->right =
	    (((DKDLeaf_p) newstree->root->leftmin->obj)->location[dimensions] +
	     ((DKDLeaf_p) sorttree->root->obj)->location[dimensions] + 1) / 2;
	else
	  /* we're at the end of the tree, just set it to virtual infinity */
	  newnode->right = MAXINT32;

	oldright = newnode->right;

	tmpstree = sorttree;
	sorttree = newstree;
	newstree = CreateSplay(dkd_compare_leaves_array[dimensions - 1]);
	DKDCopyAndFreeTreeElements(tmpstree, newstree);
      } else {
	newnode->points = 1;
	newnode->maxpoints = LOWER_NODES_MIN;
	newnode->left = oldright;
	newnode->right = MAXINT32;
/*	  ((DKDLeaf_p) sorttree->root->obj)->location[dimensions - 1];*/
	newstree = CreateSplay(dkd_compare_leaves_array[dimensions - 1]);
	SplayInsert(newstree, sorttree->root->obj);
	SplayFreeNode(sorttree->root);
	sorttree->root = NULL;
      }

      newnode->nodes =
	DKDRecurseRebuild(newnode->stree, newstree, dimensions - 1,
			  uppernodes, lowernodes, newnode->points);

      done = done + newnode->points;
      
      SplayInsert(newtree, (void *) newnode);

#ifdef DEBUG
      fflush(stdout);
      printf("Done at dimension %d\n", dimensions);
      fflush(stdout);
#endif
    }
  } else {
    /* we've already done this work, just copy over a pointer to the root
     * and be done with it.
     */
    *newtree = *sorttree;
    i = totalpoints;
  }

  free(sorttree);

  /* Return the number of nodes or leaves that were completed */
  return i;

}



void DKDRecurseCreateAndFree(SplayNode_p node, SplayTree_p newtree,
			       int dimensions)
{

  if (node == NULL) return;

  if (node->left) {
    DKDRecurseCreateAndFree(node->left, newtree, dimensions);
  }

  if (dimensions > 1) {
    DKDRecurseCreateAndFree(((DKDNode_p) node->obj)->stree->root,
			      newtree, dimensions - 1);
    free(((DKDNode_p) node->obj)->stree);
    free(node->obj);
  } else {
    /* This is a DKDLeaf_p, we don't want to free it. */
    SplayInsert(newtree, node->obj);
  }


  if (node->right) {
    DKDRecurseCreateAndFree(node->right, newtree, dimensions);
  }

  SplayFreeNode(node);
}


void DKDCreateSortTreeAndFree(DKDTree_p tree, SplayTree_p stree) {
  
  DKDRecurseCreateAndFree(tree->stree->root, stree, tree->dimensions);
  tree->stree->root = NULL;

}



/* Everything must be rebuilt cause we've done too many inserts & deletes */
/* Have to kill the entire tree and start over from scratch.              */
void DKDRebuildTree(DKDTree_p tree) {
  int32 lowernodes, uppernodes;
  SplayTree_p sorttree;

  if (tree->points > 1) {
    lowernodes = pow((double) tree->points, (double) 1.0 / tree->dimensions) *
      pow((double) log((double) tree->points) * (1.0 / M_LN2),
	  (double) 1.0 - 1.0 / tree->dimensions) + 0.5;
    if (lowernodes < LOWER_NODES_MIN)
      lowernodes = LOWER_NODES_MIN;
    uppernodes = pow((double) tree->points, 1.0 / tree->dimensions) *
      pow((double) log((double) tree->points) * (1.0 / M_LN2),
	  -1.0 / tree->dimensions) + 0.5;
  } else {
    lowernodes = LOWER_NODES_MIN;
    uppernodes = 1;
  }
#ifdef DEBUG
  fflush(stdout);
  printf("Rebuilding tree w/ %d upper nodes and %d leaves\n", uppernodes,
	 lowernodes);
  fflush(stdout);
#endif
  tree->uppernodes = uppernodes;
  tree->lowernodes = lowernodes;
  tree->nodes = uppernodes;
  tree->inserts = tree->deletes = 0;
  tree->maxmods = tree->points;
  if (tree->maxmods < LOWER_NODES_MIN / 2)
    tree->maxmods = LOWER_NODES_MIN / 2;
  sorttree = CreateSplay(dkd_compare_leaves_array[tree->dimensions-1]);
  DKDCreateSortTreeAndFree(tree, sorttree);


  DKDRecurseRebuild(tree->stree, sorttree, tree->dimensions - 1,
		    uppernodes, lowernodes, tree->points);


}



void DKDRebalanceLowerLevels(DKDNode_p upper, DKDNode_p lower,
			     int32 uppernodes, int32 lowernodes)
{
  DKDNode_p newnode;
  DKDLeaf_p splitpoint;
  SplayTree_p sorttree;
  SplayNode_p splitnode;
  int count;


  sorttree = CreateSplay(dkd_compare_leaves_array[lower->dimensions]);
  DKDRecurseCreateAndFree(lower->stree->root, sorttree, lower->dimensions);

  splitnode = DKDFindMedian(sorttree, lower->points);
  count = 0;
  SplayNode(sorttree, splitnode);
  splitnode = DKDSkipRepeats(sorttree, splitnode, lower->dimensions, &count);
  splitpoint = (DKDLeaf_p) splitnode->obj;


  /* Set up a new node to hold the new tree */
  newnode = (DKDNode_p) malloc(sizeof(DKDNode_t));
  newnode->dimensions = lower->dimensions;
  newnode->maxnodes = lower->maxnodes;
  newnode->maxpoints = lower->maxpoints;
  newnode->inserts = lower->inserts = newnode->deletes = lower->deletes = 0;

  newnode->stree = CreateSplay(dkd_compare_leaves_array[lower->dimensions]);

  SplayNode(sorttree, splitnode);

  newnode->stree->root = sorttree->root->right;
  if (newnode->stree->root) {
    newnode->stree->root->parent = NULL;
    sorttree->root->right = NULL;
  }

  free(lower->stree);
  lower->stree = sorttree;

  if (newnode->stree->root) {
    newnode->left =
      ((((DKDLeaf_p) newnode->stree->root->leftmin->obj)->location[lower->dimensions]) + splitpoint->location[lower->dimensions] + 1) / 2;
    newnode->right = lower->right;
    lower->right = newnode->left;
    newnode->points = lower->points / 2 - count;
    lower->points = lower->points - newnode->points;

    sorttree = CreateSplay(dkd_compare_leaves_array[lower->dimensions - 1]);
    DKDRecurseCreateAndFree(newnode->stree->root, sorttree, 1);

    free(newnode->stree);

    if (lower->dimensions == 1) 
      newnode->stree = CreateSplay(dkd_compare_leaves);
    else
      newnode->stree = CreateSplay(dkd_compare_nodes);
    
    newnode->nodes = DKDRecurseRebuild(newnode->stree, sorttree,
				       lower->dimensions - 1, uppernodes,
				       lowernodes, newnode->points);
  }

  sorttree = CreateSplay(dkd_compare_leaves_array[lower->dimensions - 1]);
  DKDRecurseCreateAndFree(lower->stree->root, sorttree, 1);
  free(lower->stree);

  if (lower->dimensions == 1)
    lower->stree = CreateSplay(dkd_compare_leaves);
  else
    lower->stree = CreateSplay(dkd_compare_nodes);

  lower->nodes = DKDRecurseRebuild(lower->stree, sorttree,
				   lower->dimensions - 1, uppernodes,
				   lowernodes, lower->points);
  if (newnode->stree->root) {

#ifdef DEBUG
    if (newnode->nodes != count_points(newnode->stree->root)) {
      fflush(stdout);
      fprintf(stderr, "Error, wrong number of points in rebalanced tree.\n");
    }
    if (lower->nodes != count_points(lower->stree->root)) {
      fflush(stdout);
      fprintf(stderr, "Error, wrong number of points in remainder of tree "
	      "during rebalance.\n");
    }

    if (newnode->left > newnode->right || lower->left > lower->right) {
      printf("oops again\n");
    }
#endif

    SplayInsert(upper->stree, newnode);
    upper->nodes++;
  } else {
    free(newnode->stree);
    free(newnode);
  }


}



void DKDInsert(DKDTree_p tree, dkd_vector location, void *obj) {
  int i;
  DKDNode_p curnode, newnode, oldnodes[DKD_DIMENSIONS];
  DKDNode_t curnode_t;
  DKDLeaf_t tmpleaf_t;
  DKDLeaf_p tmpleaf, newleaf;
  SplayTree_p curstree;

  
  /* I didn't want to do a loop, but this will have to change if you
   * change the dimensionality of the structure
   */
  tmpleaf_t.location[0] = location[0];
  tmpleaf_t.location[1] = location[1];
  tmpleaf_t.location[2] = location[2];
  tmpleaf_t.location[3] = location[3];
  tmpleaf = &tmpleaf_t;

#ifdef DEBUG2
  fflush(stdout);
  printf("Inserting element [%d %d %d %d]\n", location[0], location[1],
	 location[2], location[3]);
  inserts++;
#endif

  curnode = &curnode_t;
  curstree = tree->stree;
/*  tree->points++;*/
  for (i = tree->dimensions-1; i; i--) {
    curnode_t.left = curnode_t.right = location[i];
    oldnodes[i] = newnode = (DKDNode_p) SplayAccess(curstree, curnode);

#ifdef DEBUG
    if (newnode->left > curnode_t.left ||
	newnode->right < curnode_t.left) {
      fprintf(stderr, "Error inserting element\n");
    }
#endif

/*    newnode->points++;*/
    curstree = newnode->stree;
  }

  SplayAccess(curstree, tmpleaf);
  if ((!curstree->root) ||
      ((curstree->compare)(curstree->root->obj, tmpleaf))) {
    /* This is a new location, insert and update the path of nodes taken */
    newleaf = DKDCreateLeaf();
    newleaf->obj_list.next = NULL;
    newleaf->obj_list.object = obj;
    newleaf->location[0] = location[0];
    newleaf->location[1] = location[1];
    newleaf->location[2] = location[2];
    newleaf->location[3] = location[3];
    newnode->nodes++;
    newnode->points++;
    tree->points++;
    SplayInsert(curstree, newleaf);
    tree->inserts++;
    for (i=2; i<DKD_DIMENSIONS; i++) {
      (oldnodes[i]->inserts)++;
      (oldnodes[i]->points)++;
    }

    /* Check to see if I fell out of balance */
    if (tree->inserts > tree->maxmods) {
#ifdef DEBUG
      rebuild_counts++;
      printf("Rebuilding\n");
#endif
      DKDRebuildTree(tree);
    } else {
      if (oldnodes[1]->nodes >= oldnodes[1]->maxnodes) {
	if (oldnodes[2]->nodes >= oldnodes[2]->maxnodes) {
	  if (oldnodes[3]->nodes >= oldnodes[3]->maxnodes) {
#ifdef DEBUG
	    rebuild_counts++;
	    printf("Rebuilding\n");
#endif
	    DKDRebuildTree(tree);
	  } else {
	    
#ifdef DEBUG
	    rebalance_counts+=2;
	    printf("Rebalancing\n");
#endif
	    DKDRebalanceLowerLevels(oldnodes[3], oldnodes[2],
				    tree->uppernodes, tree->lowernodes);
	  }
	} else {
#ifdef DEBUG
	  rebalance_counts+=1;
	  printf("Rebalancing\n");
#endif
	  DKDRebalanceLowerLevels(oldnodes[2], oldnodes[1],
				  tree->uppernodes, tree->lowernodes);
	}
      }
#ifdef DEBUG
	fflush(stdout);
	printf("Done Rebalance\n");
#endif
    }
  } else {
    /* This is not a new location, just tack it onto the end of the list */
    /* and don't update the path w/ new inserts.                         */
    DKDObjList_p newobj;
    tmpleaf = (DKDLeaf_p) curstree->root->obj;
    newobj = (DKDObjList_p) malloc(sizeof(DKDObjList_t));
    newobj->object = obj;
    newobj->next = tmpleaf->obj_list.next;
    tmpleaf->obj_list.next = newobj;
  }
}


int DKDDeleteLeafSubset(DKDLeaf_p leaf, DKDObjList_p objs) {
  DKDObjList_p l1, l2, oldobj;
  int killit, killall = 1;

  if (objs) {
    /* Have to check to see if we want to delete first */
    oldobj = &(leaf->obj_list);
    l1 = oldobj->next;
    while (l1) {
      l2 = objs;
      killit = 0;
      while (l2 && !(killit)) {
	if (l2->object == l1->object)
	  killit = 1;
	else
	  l2 = l2->next;
      }
      if (killit) {
	free(l1->object);
	oldobj->next = l1->next;
	free(l1);
	l1 = oldobj->next;
      } else {
	killall = 0;
	l1 = l1->next;
      }
    }
    l1 = &(leaf->obj_list);
    l2 = objs;
    killit = 0;
    while (l2 && !(killit)) {
      if (l2->object == l1->object)
	killit = 1;
      else
	l2 = l2->next;
    }
    if (killit) {
      if (l1->next) {
	leaf->obj_list = *(l1->next);
	free(l1->next);
      }
    } else {
      killall = 0;
    }
  } else {
    l1 = leaf->obj_list.next;
    while (l1) {
      l2 = l1;
      l1 = l1->next;
      free(l2);
    }
  }

  return killall;
}

int DKDDelete(DKDTree_p tree, dkd_vector location, DKDObjList_p objs) {
  int i;
  DKDNode_p curnode, newnode, oldnodes[DKD_DIMENSIONS];
  DKDNode_t curnode_t;
  DKDLeaf_t tmpleaf_t;
  DKDLeaf_p tmpleaf;
  SplayTree_p curstree;
  SplayNode_p minright, maxleft;

  /* I didn't want to do a loop, but this will have to change if you
   * change the dimensionality of the structure
   */
  tmpleaf_t.location[0] = location[0];
  tmpleaf_t.location[1] = location[1];
  tmpleaf_t.location[2] = location[2];
  tmpleaf_t.location[3] = location[3];
  tmpleaf = &tmpleaf_t;
#ifdef DEBUG
  deletes++;
#endif

  curnode = &curnode_t;
  curstree = tree->stree;
  for (i = tree->dimensions-1; i; i--) {
    curnode_t.left = curnode_t.right = location[i];
    oldnodes[i] = newnode = (DKDNode_p) SplayAccess(curstree, curnode);
    curstree = newnode->stree;
  }

  SplayAccess(curstree, tmpleaf);
  if (curstree->root) {
    if ((curstree->compare)(curstree->root->obj, tmpleaf) == 0) {
      /* We have a match that we want to delete, check to see if it is
       * one of the ones we want to delete based on objs
       * (objs == NULL) deletes everything
       */
      if (DKDDeleteLeafSubset((DKDLeaf_p) curstree->root->obj, objs)) {
	/* If this returns non-zero, then everything was deleted from this
	 * leaf and everyone above needs to know
	 */
	tree->points--;
	tree->deletes++;
	for (i = 1; i < tree->dimensions; i++) {
	  (oldnodes[i]->points)--;
	  (oldnodes[i]->deletes)++;
	}
	(oldnodes[1]->nodes)--;
	DKDFreeLeaf((DKDLeaf_p) curstree->root->obj);
	SplayDelete(curstree, tmpleaf);

	if (oldnodes[1]->points == 0) {
	  if (oldnodes[2]->points == 0) {
	    oldnodes[2]->deletes++;
	    oldnodes[2]->nodes--;
	    SplayDelete(oldnodes[2]->stree, oldnodes[1]);
	    free(oldnodes[1]->stree);
	    free(oldnodes[1]);
	    DKDRebuildTree(tree);
	  } else {

#ifdef DEBUG
	    printf("Merging two lower nodes\n");
#endif
	    /* have to fix this guy up... want to merge the left and right of
	     * the two nodes to the left and right of this one so that we have
	     * full coverage over the interval, check out oldnodes[2].stree
	     *  members to try to find the correct nodes to merge.
	     */
	    minright = oldnodes[2]->stree->root->right;
	    if (minright)
	      minright = minright->leftmin;
	    maxleft = oldnodes[2]->stree->root->left;
	    if (maxleft) {
	      while(maxleft->right) {
		maxleft = maxleft->right;
	      }
	    }
	    if (minright && maxleft) {
/*
	      ((DKDNode_p) minright->obj)->left =
		(((DKDNode_p) minright->obj)->left +
		 ((DKDNode_p) maxleft->obj)->right + 1) / 2;
	      ((DKDNode_p) maxleft->obj)->right =
		((DKDNode_p) minright->obj)->left;
*/
	      ((DKDNode_p) minright->obj)->left =
		((DKDNode_p) maxleft->obj)->right =
		  (oldnodes[1]->left + oldnodes[1]->right + 1) / 2;
	    } else if (minright) {
	      ((DKDNode_p) minright->obj)->left = -MAXINT32;
	    } else if (maxleft) {
	      ((DKDNode_p) maxleft->obj)->right = MAXINT32;
	    } else {
	      /* I shouldn't have hit this region... */
	      printf("Clean up needed...fucked up\n");
	    }

	    SplayDelete(oldnodes[2]->stree, oldnodes[1]);
	    oldnodes[2]->deletes++;
	    oldnodes[2]->nodes--;
	    free(oldnodes[1]->stree);
	    free(oldnodes[1]);
	  }
        }


	if (tree->deletes > tree->maxmods) {
#ifdef DEBUG
	  rebuild_counts++;
	  printf("Rebuilding\n");
#endif
	  DKDRebuildTree(tree);
	}
	
	return 0;
	
      }
      /* I deleted some of the objects, but not all of them */
      return 0;
    }
  }
  
  return 1;

}



DKDLeaf_p DKDAccess(DKDTree_p tree, DKDLeaf_p node) {
  int i;
  DKDLeaf_p leaf;
  DKDNode_p curnode, newnode;
  DKDNode_t curnode_t;
  SplayTree_p curstree;
  
  curnode = &curnode_t;
  curstree = tree->stree;
  for (i = tree->dimensions-1; i; i--) {
    curnode_t.left = curnode_t.right = node->location[i];
    newnode = (DKDNode_p) SplayAccess(curstree, curnode);
    curstree = (SplayTree_p) newnode->stree;
  }

  leaf = (DKDLeaf_p) SplayAccess(curstree, node);

  return leaf;

}
