import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.xml.sax.SAXException;

/**
 * Compile, execute and shows results of an XPath expression applied to user specified XML file.
 * Prompt user if a timed test of the expression should be averaged
 **/

public class XPathPerformanceTester {

    public static final String DEFAULT_XML_FILENAME = "entryLevel_Big.xml";
    public static final int DEFAULT_NUM_TESTS = 10;
    public static final String PREDICATE_LAST = "LAST()";
    public static final String WALK_INDICATOR = "~";


    // Evaluate predicate by "Walking The Dom"
    public static Node walkPredicateEval(Document domDoc, String expr) {
        if ((expr == null) || (expr.length() < 1)) {
            System.out.println("Null expression passed to walkPredicateEval()");
            return null;
        }

        // Isolate element to search for from predicate
        int elementPos = expr.indexOf('[');
        if (elementPos < 0) {
            System.out.println("Invalid 'Walk the DOM' expression!"+
                    "   Must be '~' <element_name> '[' <predicate_value> ']'" +
                    "   ==>  Example:  ~Entry[last()]");
            return null;
        }
        String element = expr.substring(0, elementPos);
        String predicate = expr.substring(elementPos + 1);
        if (predicate.endsWith("]")) {
            predicate = predicate.substring(0, predicate.length() - 1); // remove ]
        }
        // Get the root
        // Walk the DOM to satisfy expression
        Element rootElement = domDoc.getDocumentElement();
        Node node = null;
        NodeList modelNodeList = rootElement.getElementsByTagName(element);
        if (modelNodeList != null && modelNodeList.getLength() > 0) {
            int totalElements = modelNodeList.getLength();
            int elementToRetrieve = 0;
            // Convert the predicate to an integer
            if (predicate.toUpperCase().equals(PREDICATE_LAST)) {
                elementToRetrieve = totalElements - 1;
            } else {
                // findPos had better be an integer
                try {
                    elementToRetrieve = Integer.parseInt(predicate);
                    elementToRetrieve--; // Adjust for 0 based index
                } catch (NumberFormatException e) {
                    System.out.println("Predicate value (" + predicate + ") must be an integer or 'last()'");
                    return null;
                }
            }
            node = modelNodeList.item(elementToRetrieve);
        }
        return node;
    }


    public static void main(String... args) {
        // Setup an InputStreamReader to read from the keyboard
        InputStreamReader reader = new InputStreamReader(System.in);
        BufferedReader in = new BufferedReader(reader);

        // Instantiate the factory that supplies the DOM parser
        DocumentBuilderFactory builderFactory =
                DocumentBuilderFactory.newInstance();

        DocumentBuilder domParser = null;
        try {
            // Instantiate the DOM parser
            domParser = builderFactory.newDocumentBuilder();

            // Load the DOM Document from the XML data using the parser
            Document domDocument =
                    domParser.parse(getFileInputStreamName(in));

            // Instantiate an XPath object which compiles
            // and evaluates XPath expressions.
            XPath xPath = XPathFactory.newInstance().newXPath();

            while (true) {
                System.out.print("Enter expression (blank line to exit): ");
                String expr = in.readLine(); // Holds the XPath expression

                try {
                    if ((expr == null) || (expr.length() == 0)) {
                        System.exit(0); // User is done entering expressions
                    }

                    if (expr.startsWith(WALK_INDICATOR)) {
                        // Evaluate predicate expression by 'Walking the DOM'
                        if (walkPredicateEval(domDocument, expr.substring(1)) != null) {
                            // Expression evaluated successfully, get an average execution time over DEFAULT_NUM_TESTS
                            long walkTimes = 0;
                            for (int xxx = 0; xxx < DEFAULT_NUM_TESTS; xxx++) {
                                long startTime = System.currentTimeMillis();
                                // 'Walk the DOM' but throw away nodes returned
                                walkPredicateEval(domDocument, expr.substring(1));
                                walkTimes += System.currentTimeMillis() - startTime;
                            }
                            walkTimes /= DEFAULT_NUM_TESTS;
                            System.out.println(DEFAULT_NUM_TESTS + " tests performed." +
                                    " Average time to 'Walk the DOM' for '" +
                                    expr.substring(1) + "' is " + walkTimes);
                        }
                        continue;
                    } else {
                        // Compile and evaluate using XPath
                        Node resNode = null;
                        long start_time = System.currentTimeMillis();

                        NodeList resNodeList = (NodeList) xPath.compile(expr).evaluate(domDocument,
                                XPathConstants.NODESET);
                        long end_time = System.currentTimeMillis();
                        long difference = end_time - start_time;
                        System.out.println("Milliseconds taken to evaluate " +
                                expr + " was " + difference);
                        if (resNodeList != null) {
                            int lenList = resNodeList.getLength();
                            System.out.println("Number of nodes in NodeList: " +
                                    lenList);
                            if (lenList > 5) {
                                System.out.println("Showing first 5 nodes...");
                                lenList = 5;
                            }
                            for (int i = 1; i <= lenList; i++) {
                                resNode = resNodeList.item(i - 1);
                                String resNodeNameStr = resNode.getNodeName();
                                String resNodeTextStr = resNode.getTextContent();
                                // Look for non-blank lines
                                boolean fIsBlank = true; // Assume it is one or more blank lines
                                for (int j = 0; j < resNodeTextStr.length(); j++) {
                                    char c = resNodeTextStr.charAt(j);
                                    if (!Character.isWhitespace(c) && !Character.isLetter(c)) {
                                        fIsBlank = false;
                                        break;
                                    }
                                }
                                if (fIsBlank) {
                                    // Count the number of LineFeed characters
                                    int lines = 1;
                                    int pos = 0;
                                    while ((pos = resNodeTextStr.indexOf("\n", pos) + 1) != 0) {
                                        lines++;
                                    }
                                    resNodeTextStr = "<< " + lines + " blank lines >>";
                                }
                                System.out.println(i + ": " + resNode + "  (NodeName:'" +
                                        resNodeNameStr + "'    NodeTextContent:'" +
                                        resNodeTextStr + "')");
                            }
                            System.out.print("    >>> Enter 'y' to get average time for " +
                                    DEFAULT_NUM_TESTS + " tests: ");
                            String response = null; // Holds the response
                            try {
                                response = in.readLine();
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                            if ((response != null) && (response.length() > 0) &&
                                    response.substring(0, 1).toUpperCase().equals("Y")) {

                                long[] times = new long[DEFAULT_NUM_TESTS];
                                for (int xxx = 0; xxx < DEFAULT_NUM_TESTS; xxx++) {
                                    long startTime = System.currentTimeMillis();
                                    NodeList testNodeList = (NodeList) xPath.compile(expr).
                                            evaluate(domDocument, XPathConstants.NODESET);
                                    times[xxx] = System.currentTimeMillis() - startTime;
                                    System.out.print(times[xxx] + "   ");
                                    if (((xxx + 1) % 25) == 0) {
                                        System.out.println();
                                    }
                                }
                                long average = 0;
                                for (int yyy = 0; yyy < DEFAULT_NUM_TESTS; yyy++) {
                                    average += times[yyy];
                                }
                                average /= DEFAULT_NUM_TESTS;
                                System.out.println("\nAverage milliseconds taken to evaluate " +
                                        expr + " was " + average);
                            }
                        }
                    }

                } catch (XPathExpressionException e) {
                    System.out.println("XPathExpression Exception!!!");
                    System.out.println(e.getMessage());
                }

                outputSeparator();

            } // end  while (true)

        } catch (SAXException e) {
            // Even though we are using a DOM parser a SAXException is thrown
            // if the DocumentBuilder cannot parse the XML file
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        } catch (ParserConfigurationException e) {
            e.printStackTrace();
        }
    }

    // Helper method to load the XML file into the DOM Document
    public static String getFileInputStreamName(BufferedReader inputReader) {
        System.out.print("Enter XML file name (abc.xml) or leave blank to use " + DEFAULT_XML_FILENAME + ": ");
        String fileName = null;
        try {
            fileName = inputReader.readLine();
        } catch (IOException e) {
            e.printStackTrace();
        }
        if ((fileName == null) || (fileName.length() == 0)) {
            return DEFAULT_XML_FILENAME;
        }
        return fileName;
    }

    // Helper method to pretty up the output
    public static void outputSeparator() {
        System.out.println("=+=+=+=+=+=+=+=+");
    }

}