package com.javacodegeeks.nio.async_channels_tutorial.file;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.apache.commons.lang3.StringUtils;

import com.javacodegeeks.nio.async_channels_tutorial.Constants;

public final class FileChannel {

    private static final long START_POS = 0;
    private static final String UNABLE_TO_READ_CONTENTS = "Unable to read file contents";
    private static final String UNABLE_TO_WRITE_CONTENTS = "Unable to write to file";

    public String read(final String path) {
	final Path pathToFile = Paths.get(path);
	String result = StringUtils.EMPTY;

	try (AsynchronousFileChannel channel = AsynchronousFileChannel.open(pathToFile, StandardOpenOption.READ)) {
	    result = read(channel, ByteBuffer.allocate(Constants.BUFFER_SIZE), new StringBuilder(), START_POS);
	} catch (IOException e) {
	    throw new RuntimeException(UNABLE_TO_READ_CONTENTS, e);
	}

	return result;
    }

    public void write(final String path, final String contents) {
	final Path pathToFile = Paths.get(path);

	try (AsynchronousFileChannel channel = AsynchronousFileChannel.open(pathToFile, StandardOpenOption.WRITE, StandardOpenOption.CREATE)) {
	    final ByteBuffer buffer = ByteBuffer.wrap(contents.getBytes());

	    write(channel, buffer, START_POS);
	} catch (IOException e) {
	    throw new RuntimeException(UNABLE_TO_WRITE_CONTENTS, e);
	}
    }

    private void write(final AsynchronousFileChannel channel, final ByteBuffer buffer, final long filePosition) {
	assert !Objects.isNull(channel) && !Objects.isNull(buffer);

	final Future<Integer> result = channel.write(buffer, filePosition);
	try {
	    final int bytesWritten = result.get();
	    while (buffer.hasRemaining()) {
		buffer.compact();
		write(channel, buffer, bytesWritten + filePosition);
	    }
	} catch (InterruptedException | ExecutionException e) {
	    throw new RuntimeException(UNABLE_TO_WRITE_CONTENTS, e);
	}
    }

    private String read(final AsynchronousFileChannel channel, final ByteBuffer buffer, final StringBuilder contents, final long filePosition) {
	assert !Objects.isNull(channel) && !Objects.isNull(buffer) && !Objects.isNull(contents);

	final Future<Integer> result = channel.read(buffer, filePosition);
	try {
	    final int bytesRead = result.get();
	    if (bytesRead != -1) {
		contents.append(new String(buffer.array()).trim());

		buffer.clear();
		return read(channel, buffer, contents, filePosition + bytesRead);
	    } else {
		return contents.toString();
	    }
	} catch (InterruptedException | ExecutionException e) {
	    throw new RuntimeException(UNABLE_TO_READ_CONTENTS, e);
	}
    }

}
