package com.javacodegeeks.nio.async_channels_tutorial.server;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousChannelGroup;
import java.nio.channels.AsynchronousSocketChannel;
import java.nio.channels.CompletionHandler;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.commons.lang3.StringUtils;

import com.javacodegeeks.nio.async_channels_tutorial.Constants;
import com.javacodegeeks.nio.async_channels_tutorial.IdGenerator;

public final class Client implements AsyncNIOMessageRelayParticipant {

    private final ConcurrentMap<String, StringBuilder> messageCache;
    private final AsynchronousChannelGroup group;
    private final InetSocketAddress remoteAddress;
    private final int numConnections;
    private final CountDownLatch latch;
    private final boolean stopCalled = false;
    private final Lock lock = new ReentrantLock();

    public Client(final int numConnections, final int port, final int poolSize) {
	this.numConnections = numConnections;
	this.messageCache = new ConcurrentHashMap<>(numConnections);
	this.remoteAddress = new InetSocketAddress(port);
	this.latch = new CountDownLatch(numConnections);

	try {
	    this.group = AsynchronousChannelGroup.withThreadPool(Executors.newFixedThreadPool(poolSize));
	} catch (IOException e) {
	    throw new IllegalStateException("Client: Unable to initialize thread group pool for client");
	}
    }

    @Override
    public void start() {
	for (int i = 0; i < this.numConnections; i++) {
	    AsynchronousSocketChannel client;
	    try {
		client = AsynchronousSocketChannel.open(this.group);
		connect(client, IdGenerator.generate());
	    } catch (IOException e) {
		throw new RuntimeException("Client: Unable to start clients", e);
	    }
	}

	try {
	    this.latch.await();
	    stop();
	} catch (InterruptedException e) {
	    throw new RuntimeException("Client: Unable to wait for requests to finish");
	}
    }

    @Override
    public void stop() {
	if (!this.stopCalled) {
	    try {
		this.lock.lock();
		if (!this.stopCalled) {
		    stopChannelGroup(this.group);
		}
	    } finally {
		this.lock.unlock();
	    }
	}
    }

    public Map<String, StringBuilder> getMessageCache() {
	return Collections.unmodifiableMap(this.messageCache);
    }

    private void read(final AsynchronousSocketChannel channel, final String requestId) {
	assert !Objects.isNull(channel);

	final ByteBuffer buffer = create(Constants.BUFFER_SIZE);
	channel.read(buffer, requestId, new CompletionHandler<Integer, String>() {

	    @Override
	    public void completed(final Integer result, final String attachment) {
		System.out.println(String.format("Client: Read Completed in thread %s for request %s", Thread.currentThread().getName(), attachment));

		// Extract what was read from the buffer.
		final String message = Client.this.extract(buffer);

		// Update the cache with the message.
		Client.this.updateMessageCache(attachment, message, Client.this.messageCache);

		if (message.contains(Constants.END_MESSAGE_MARKER) || result == -1) {
		    extractEndMessageMarker(attachment);

		    closeChannel(channel);
		    Client.this.latch.countDown();
		} else {
		    read(channel, attachment);
		}
	    }

	    @Override
	    public void failed(final Throwable exc, final String attachment) {
		System.out.println(String.format("Client: Read Failed in thread %s", Thread.currentThread().getName()));
		exc.printStackTrace();

		Client.this.latch.countDown();
		closeChannel(channel);
	    }

	    private void extractEndMessageMarker(final String requestId) {
		assert !Objects.isNull(requestId);

		final String message = Client.this.messageCache.get(requestId).toString();
		Client.this.messageCache.put(requestId, new StringBuilder(message.replace(Constants.END_MESSAGE_MARKER, StringUtils.EMPTY)));

	    }
	});
    }

    private void write(final AsynchronousSocketChannel channel, final String requestId) {
	assert !Objects.isNull(channel);

	final ByteBuffer contents = create(Constants.BUFFER_SIZE);
	contents.put(requestId.getBytes());
	contents.put(Constants.END_MESSAGE_MARKER.getBytes());
	contents.flip();

	channel.write(contents, requestId, new CompletionHandler<Integer, String>() {

	    @Override
	    public void completed(final Integer result, final String attachment) {
		System.out.println(String.format("Client: Write Completed in thread %s", Thread.currentThread().getName()));
		read(channel, attachment);
	    }

	    @Override
	    public void failed(final Throwable exc, final String attachment) {
		System.out.println(String.format("Client: Write Failed in thread %s", Thread.currentThread().getName()));
		exc.printStackTrace();

		Client.this.latch.countDown();
		closeChannel(channel);
	    }
	});
    }

    private void connect(final AsynchronousSocketChannel channel, final String requestId) {
	channel.connect(this.remoteAddress, requestId, new CompletionHandler<Void, String>() {

	    @Override
	    public void completed(final Void result, final String attachment) {
		System.out.println(String.format("Client: Connect Completed in thread %s", Thread.currentThread().getName()));
		updateMessageCache(attachment, StringUtils.EMPTY, Client.this.messageCache);

		write(channel, attachment);
	    }

	    @Override
	    public void failed(final Throwable exc, final String attachment) {
		System.out.println(String.format("Client: Connect Failed in thread %s", Thread.currentThread().getName()));
		exc.printStackTrace();

		Client.this.latch.countDown();
		closeChannel(channel);
	    }
	});
    }
}
