package com.javacodegeeks.nio.async_channels_tutorial.server;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousChannelGroup;
import java.nio.channels.AsynchronousServerSocketChannel;
import java.nio.channels.AsynchronousSocketChannel;
import java.nio.channels.CompletionHandler;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;

import org.apache.commons.lang3.StringUtils;

import com.javacodegeeks.nio.async_channels_tutorial.Constants;
import com.javacodegeeks.nio.async_channels_tutorial.IdGenerator;

public final class Server implements AsyncNIOMessageRelayParticipant {

    private final AsynchronousServerSocketChannel server;
    private final AsynchronousChannelGroup group;
    private final ConcurrentMap<String, StringBuilder> messageCache;
    private final String echo;

    public Server(final int port, final int poolSize, final String echo) {
	try {
	    this.group = AsynchronousChannelGroup.withThreadPool(Executors.newFixedThreadPool(poolSize));
	    this.server = AsynchronousServerSocketChannel.open(this.group).bind(new InetSocketAddress(port));
	    this.messageCache = new ConcurrentHashMap<>();
	    this.echo = echo;
	} catch (IOException e) {
	    throw new IllegalStateException("Server: Unable to start server", e);
	}
    }

    @Override
    public void start() {
	accept(IdGenerator.generate());
    }

    @Override
    public void stop() {
	stopChannelGroup(this.group);
    }

    private void read(final AsynchronousSocketChannel channel, final String requestId) {
	assert !Objects.isNull(channel);

	final ByteBuffer buffer = create(Constants.BUFFER_SIZE);
	channel.read(buffer, requestId, new CompletionHandler<Integer, String>() {

	    @Override
	    public void completed(final Integer result, final String attachment) {
		System.out.println(String.format("Server: Read Completed in thread %s", Thread.currentThread().getName()));

		// Extract what was read from the buffer.
		final String message = Server.this.extract(buffer);

		// Update the cache with the message.
		Server.this.updateMessageCache(attachment, message, Server.this.messageCache);

		if (result != -1 && message.contains(Constants.END_MESSAGE_MARKER)) {

		    // Create echo message.
		    final String echo = createEcho(attachment);

		    // Send echo message to client.
		    doEcho(echo, channel);

		    // Prepare to read again.
		    read(channel, attachment);
		} else if (result != -1 && !message.contains(Constants.END_MESSAGE_MARKER)) {
		    // Prepare to read again.
		    read(channel, attachment);
		} else {
		    closeChannel(channel);
		}
	    }

	    @Override
	    public void failed(final Throwable exc, final String attachment) {
		System.out.println(String.format("Server: Read Failed in thread %s", Thread.currentThread().getName()));
		exc.printStackTrace();

		closeChannel(channel);
	    }

	    private String createEcho(final String requestId) {
		assert StringUtils.isNotEmpty(requestId);

		final String fullResponse = Server.this.echo + ":" + Server.this.messageCache.get(requestId).toString().replace(Constants.END_MESSAGE_MARKER, StringUtils.EMPTY);
		return fullResponse + Constants.END_MESSAGE_MARKER;
	    }

	    private void doEcho(final String response, final AsynchronousSocketChannel channel) {
		assert !Objects.isNull(channel) && StringUtils.isNotEmpty(response);

		final ByteBuffer responseBuffer = create(Constants.BUFFER_SIZE);
		responseBuffer.put(response.toString().trim().getBytes());

		responseBuffer.flip();

		channel.write(responseBuffer);
	    }
	});
    }

    private void accept(final String requestKey) {
	assert !Objects.isNull(requestKey);

	this.server.accept(requestKey, new CompletionHandler<AsynchronousSocketChannel, String>() {
	    public void completed(final AsynchronousSocketChannel channel, final String attachment) {

		// Delegate off to another thread for the next connection.
		accept(IdGenerator.generate());

		// Delegate off to another thread to handle this connection.
		Server.this.read(channel, attachment);
	    }

	    public void failed(final Throwable exc, final String attachment) {
		System.out.println(String.format("Server: Failed to accept connection in thread %s", Thread.currentThread().getName()));
		exc.printStackTrace();
	    }
	});
    }
}
