package com.javacodegeeks.nio.async_channels_tutorial.file;

import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.IOException;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import com.javacodegeeks.nio.async_channels_tutorial.AbstractTest;

public class LockTest extends AbstractTest {

    private String filePath = "/tmp/async-file.txt";

    @Before
    public void setUp() throws IOException {
	final File file = new File(this.filePath);
	if (!file.exists()) {
	    file.createNewFile();
	}
    }

    @After
    public void tearDown() {
	final File file = new File(this.filePath);
	if (file.exists() && !file.isDirectory()) {
	    file.delete();
	}
    }

    @Test
    public void testExclusiveLock() throws IOException, InterruptedException, ExecutionException {
	try (AsynchronousFileChannel channel = AsynchronousFileChannel.open(Paths.get(this.filePath), StandardOpenOption.WRITE, StandardOpenOption.CREATE)) {
	    final FileLock lock = channel.lock().get();

	    assertTrue("Lock is not exclusive", !lock.isShared());
	}
    }

    @Test
    public void testSharedLock() throws IOException, InterruptedException, ExecutionException {
	try (AsynchronousFileChannel channel = AsynchronousFileChannel.open(Paths.get(this.filePath), StandardOpenOption.READ, StandardOpenOption.CREATE)) {
	    final FileLock lock = channel.lock(0, 0L, true).get();

	    assertTrue("Lock is exclusive", lock.isShared());
	}
    }

    @Test(expected = OverlappingFileLockException.class)
    public void testOverlappingLock() {
	final CountDownLatch innerThreadLatch = new CountDownLatch(1);
	final CountDownLatch testThreadLatch = new CountDownLatch(1);

	try (AsynchronousFileChannel channel = AsynchronousFileChannel.open(Paths.get(this.filePath), StandardOpenOption.WRITE, StandardOpenOption.CREATE)) {

	    new Thread() {
		public void run() {
		    try {
			channel.lock().get();
			innerThreadLatch.countDown();
			testThreadLatch.await();
		    } catch (OverlappingFileLockException | ExecutionException | InterruptedException e) {
			throw new RuntimeException("Unable to get lock on file for overlapping lock test", e);
		    }
		}
	    }.start();

	    innerThreadLatch.await();
	    channel.lock().get();
	} catch (InterruptedException | ExecutionException | IOException e) {
	    throw new RuntimeException(e);
	} finally {
	    testThreadLatch.countDown();
	}
    }
}
