package com.javacodegeeks.java9_regex_example;

import java.util.Arrays;
import java.util.List;
import java.util.regex.MatchResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

final class Misc {

    private Misc() {
        throw new IllegalStateException("Instantiation not allowed");
    }

    static void run() {
        literal();
        replacement();
        matchResult();
        characterClass();
        negateCharacterClass();
        union();
        mixedUnion();
        intersection();
        inRange();
        replacing();
        streaming();
        predicated();
        splitPattern();
        boundary();
    }

    private static void literal() {
        System.out.println("Misc - literal\n");

        final String input = "oxoxox";

        // Simple pattern match on literal String value
        final Pattern pattern = Pattern.compile("x");
        final Matcher matcher = pattern.matcher(input);

        // Should match 3 'x' values at progressively increasing indexes.
        printResults(matcher, input, pattern.pattern());
        System.out.println("-----\n");
    }

    private static void replacement() {
        System.out.println("Misc - replacement\n");

        final String input = "this and that and those and these";

        final Pattern pattern = Pattern.compile("and");
        final Matcher matcher = pattern.matcher(input);
        final StringBuilder sb = new StringBuilder();

        while (matcher.find()) {
            matcher.appendReplacement(sb, "or");
        }

        matcher.appendTail(sb);
        System.out.printf("\t%s -> %s\n", input, sb.toString());

        System.out.println("-----\n");
    }

    private static void matchResult() {
        System.out.println("Misc - match result\n");

        final String input = "oneoneonetwoone";

        final Pattern pattern = Pattern.compile("one");
        final Matcher matcher = pattern.matcher(input);

        while (matcher.find()) {
            // Creates a read only snapshot of the result.
            final MatchResult result = matcher.toMatchResult();

            System.out.printf("\tStart index of match %d\n", result.start());
            System.out.printf("\tEnd index of match %d\n", result.end());
            System.out.printf("\tValue is %s\n\n", result.group());
        }

        System.out.println("-----\n");
    }

    private static void characterClass() {
        System.out.println("Misc - character class\n");

        final String input = "At 10:00am I have Computer science class and at 11:00am I have a hall pass and at 12:00pm I have no class";

        // Ensure we capture any word(s) ending is 'ass' but starting with a 'c'
        // followed by a 'l'
        final Pattern pattern = Pattern.compile("[c][l]ass");
        final Matcher matcher = pattern.matcher(input);

        printResults(matcher, input, pattern.pattern());
        System.out.println("-----\n");
    }

    private static void negateCharacterClass() {
        System.out.println("Misc - negate character class\n");

        final String input = "At 10:00am I have Computer science class and at 11:00am I have a hall pass and at 12:00pm I have no class and at 4:00pm we leave in mass";

        // Here we negate the first two characters of the pattern we want matched by
        // saying all word(s) that end with 'ass'
        // but that don't start with the following characters 'cl'
        final Pattern pattern = Pattern.compile("[^c][^l]ass");
        final Matcher matcher = pattern.matcher(input);

        printResults(matcher, input, pattern.pattern());
        System.out.println("-----\n");
    }

    private static void union() {
        System.out.println("Misc - union\n");

        final String input = "abcdefghijklmnopqrstuvwxyz";

        // We are interested in the letters 'a' through to 'd' and 'u' through to 'z'
        // all inclusive.
        final Pattern pattern = Pattern.compile("[a-d[u-z]]");
        final Matcher matcher = pattern.matcher(input);

        printResults(matcher, input, pattern.pattern());
        System.out.println("-----\n");
    }

    private static void mixedUnion() {
        System.out.println("Misc - mixed union\n");

        final String input = "abcdefghijklmnopqrstuvwxyz123456789";

        // A union set comprising letters and numbers.
        final Pattern pattern = Pattern.compile("[a-d[4-8]]");
        final Matcher matcher = pattern.matcher(input);

        printResults(matcher, input, pattern.pattern());
        System.out.println("-----\n");
    }

    private static void intersection() {
        System.out.println("Misc - intersection\n");

        final String input = "abcdefghijklmnopqrstuvwxyz";

        // We are interested in the overlap / intersection of the character classes 'a'
        // through 'd' and the letters 'b',c',c','y'
        // meaning we will only get 'b' and 'c'
        final Pattern pattern = Pattern.compile("[a-d&&[bcxyz]]");
        final Matcher matcher = pattern.matcher(input);

        printResults(matcher, input, pattern.pattern());
        System.out.println("-----\n");
    }

    private static void inRange() {
        System.out.println("Misc - in range\n");

        final String input = "bow flow row now snow crow mow vow";

        // Here we are interested in whole words '\\b' that end in 'ow' but that start
        // with either 'b' or 'n'.
        // ie: 'bow' or 'now'
        final Pattern pattern = Pattern.compile("(\\b[bn]ow)\\b");
        final Matcher matcher = pattern.matcher(input);

        printResults(matcher, input, pattern.pattern());
        System.out.println("-----\n");
    }

    private static void replacing() {
        System.out.println("Misc - replacing\n");

        final String input = "java is a coffee, java is an island, java is also a programming language.";

        final Pattern pattern = Pattern.compile("java");
        final Matcher matcher = pattern.matcher(input);

        System.out.printf("\t The resulting string is %s\n", matcher.replaceAll((result) -> result.group().toUpperCase()));
        System.out.println("-----\n");
    }

    private static void streaming() {
        System.out.println("Misc - streaming\n");

        final String input = "java is a coffee, java is an island, java is also a programming language.";

        final Pattern pattern = Pattern.compile("java");
        final Matcher matcher = pattern.matcher(input);

        final Stream<MatchResult> results = matcher.results();
        results.forEach((result) -> {
            System.out.printf("\tStart index of match %d\n", result.start());
            System.out.printf("\tEnd index of match %d\n", result.end());
            System.out.printf("\tValue is %s\n", result.group());

            if (result.end() < matcher.regionEnd()) {
                System.out.println("");
            }
        });

        System.out.println("-----\n");
    }

    private static void predicated() {
        System.out.println("Misc - predicated\n");

        final List<String> input = Arrays.asList("bow", "flow", "row", "now", "snow", "crow", "mow", "vow");
        final Pattern pattern = Pattern.compile("(\\b[bn]ow)\\b");

        // Demonstrating the use of 'pattern.asPredicate'
        input.stream().filter(pattern.asPredicate()).forEach(elem -> {
            System.out.printf("\t %s\n", elem);
        });

        System.out.println("");
        System.out.println("-----\n");
    }

    private static void splitPattern() {
        System.out.println("Misc - split pattern\n");

        final String input = "java is a coffee, java is an island, java is also a programming language.";

        final Pattern pattern = Pattern.compile("java");

        // Very conveniently we can split the pattern based on it's regular expression
        // and the result will
        // be stored in an array.
        // In this case the result will be:
        /*
         * is a coffee, is an island, is also a programming language.
         */
        final String[] result = pattern.split(input);

        for (String elem : result) {
            System.out.printf("\t %s\n", elem);
        }

        System.out.println("");
        System.out.println("-----\n");
    }

    private static void boundary() {
        System.out.println("Misc - boundary\n");

        final String input = "dog hotdog";

        // Demonstrating the word boundary pattern.
        // In a typical pattern without word boundaries we would match 'dog' twice, once
        // as the word itself, then the other as a part of 'hotdog' but not f we specify
        // whole words
        final Pattern pattern = Pattern.compile("\\bdog\\b");
        final Matcher matcher = pattern.matcher(input);

        printResults(matcher, input, pattern.pattern());
        System.out.println("-----\n");
    }

    private static void printResults(final Matcher matcher, final String input, final String regex) {

        System.out.printf("\tInput is %s\n", input);
        System.out.printf("\tRegex is %s\n\n", regex);

        while (matcher.find()) {

            System.out.printf("\tStart index of match %d\n", matcher.start());
            System.out.printf("\tEnd index of match %d\n", matcher.end());
            System.out.printf("\tValue is %s\n", matcher.group());

            if (matcher.end() < matcher.regionEnd()) {
                System.out.println("");
            }
        }
    }
}
