package org.jcg.zheng;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.Arrays;

import org.junit.Test;

public class CopyObjectArrayTest extends TestBase {

	private int length = 3;

	private DemoObject maryObject = new DemoObject('A', 1, "Mary");
	private DemoObject someObject = new DemoObject('B', 2, "Some data for test");

	private DemoObject[] sourceObjectArray = { maryObject, someObject, null };

	private void validate_deepCopy_ObjectArray(DemoObject[] clonedArray) {
		// the copied object is a different reference
		assertFalse(System.identityHashCode(sourceObjectArray) - System.identityHashCode(clonedArray) == 0);

		assertEquals(3, clonedArray.length);

		// the object element reference is different
		assertFalse(System.identityHashCode(maryObject) - System.identityHashCode(clonedArray[0]) == 0);
		assertFalse(System.identityHashCode(someObject) - System.identityHashCode(clonedArray[1]) == 0);

		assertEquals(maryObject, clonedArray[0]);
		assertEquals(someObject, clonedArray[1]);

		// the object change will NOT affect the other because of deep copy
		clonedArray[1].setName("Changed");
		assertEquals("Some data for test", sourceObjectArray[1].getName());

		sourceObjectArray[1].setNumber(99);
		assertEquals(2, clonedArray[1].getNumber());
	}

	private void validate_shallowCopy_ObjectArray(DemoObject[] clonedArray) {
		// the copied object is a different reference
		assertFalse(System.identityHashCode(sourceObjectArray) - System.identityHashCode(clonedArray) == 0);

		// the copied object element is the same reference
		assertTrue(System.identityHashCode(maryObject) - System.identityHashCode(clonedArray[0]) == 0);
		assertTrue(System.identityHashCode(someObject) - System.identityHashCode(clonedArray[1]) == 0);

		assertEquals(maryObject, clonedArray[0]);
		assertEquals(someObject, clonedArray[1]);

		// for the copied object, change one will affect the other due to object
		// reference is same
		clonedArray[1].setName("Changed");
		assertEquals("Changed", sourceObjectArray[1].getName());

		sourceObjectArray[1].setNumber(99);
		assertEquals(99, clonedArray[1].getNumber());

	}

	@Test
	public void via_Arrays_copyof() {
		DemoObject[] clonedArray = Arrays.copyOf(sourceObjectArray, sourceObjectArray.length);

		assertEquals(sourceObjectArray.length, clonedArray.length);
		validate_shallowCopy_ObjectArray(clonedArray);
	}

	@Test
	public void via_Arrays_copyOfRange() {
		// since Java 1.6
		DemoObject[] clonedArray = Arrays.copyOfRange(sourceObjectArray, copyStartAtZeroIndex,
				sourceObjectArray.length - 1);

		assertEquals(sourceObjectArray.length - 1, clonedArray.length);
		validate_shallowCopy_ObjectArray(clonedArray);
	}

	@Test
	public void via_Arrays_copyOfRange_padding_null_when_dest_is_larger_then_original() {
		// since Java 1.6
		DemoObject[] clonedArray = Arrays.copyOfRange(sourceObjectArray, copyStartAtZeroIndex,
				sourceObjectArray.length + 1);

		assertEquals(sourceObjectArray.length + 1, clonedArray.length);
		assertNull(clonedArray[sourceObjectArray.length]);
		validate_shallowCopy_ObjectArray(clonedArray);
	}

	@Test
	public void via_Arrays_stream_toArray() {
		DemoObject[] clonedArray = Arrays.stream(sourceObjectArray).map(DemoObject::new).toArray(DemoObject[]::new);

		validate_deepCopy_ObjectArray(clonedArray);
	}

	@Test
	public void via_Object_clone() {
		// Since Java 1.0
		DemoObject[] clonedArray = sourceObjectArray.clone();

		assertEquals(sourceObjectArray.length, clonedArray.length);
		validate_shallowCopy_ObjectArray(clonedArray);
	}

	@Test
	public void via_System_arraycopy() {

		DemoObject[] destination = new DemoObject[length];
		// Since Java 1.0
		System.arraycopy(sourceObjectArray, copyStartAtZeroIndex, destination, copyStartAtZeroIndex, length);

		assertEquals(length, destination.length);
		validate_shallowCopy_ObjectArray(destination);
	}

	@Test
	public void via_System_arraycopy_padding_null_when_destination_is_bigger_than_original() {
		DemoObject[] destination = new DemoObject[5];

		// Since Java 1.0
		System.arraycopy(sourceObjectArray, copyStartAtZeroIndex, destination, copyStartAtZeroIndex,
				sourceObjectArray.length);

		assertNull(destination[sourceObjectArray.length]);
		assertNull(destination[sourceObjectArray.length + 1]);
	}

}