package com.springboot.pagination.controller;

import com.springboot.pagination.dto.ResponseDto;
import com.springboot.pagination.model.Movie;
import com.springboot.pagination.service.MovieService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Collections;
import java.util.List;

// Useful to create the Restful microservices.
@RestController
@RequestMapping("/api/movies")
public class MovieController {

    private static final Logger LOGGER = LoggerFactory.getLogger(MovieController.class);

    // @Autowired annotation provides the automatic dependency injection.
    @Autowired
    MovieService movieService;

    // @GetMapping annotation handles the http get request matched with the given uri.
    // @RequestParam annotation reads the request parameters (i.e. basically is a query string)
    // Sample urls -
    // 1. http://localhost:8089/api/movies/getAll
    // 2. http://localhost:8089/api/movies/getAll?pageNumber=2&pageSize=10
    // 3. http://localhost:8089/api/movies/getAll?pageSize=15&pageNumber=2
    @GetMapping(value = "/getAll", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<ResponseDto> getAllMovies(
            @RequestParam(name = "pageNumber", defaultValue = "0") final int pageNumber,    // In spring the default page number starts with '0'.
            @RequestParam(name = "pageSize", defaultValue = "5") final int pageSize) {
        LOGGER.info("Getting all the movies from the database for page-number= {} and page-size= {}.", pageNumber,
                pageSize);
        final ResponseEntity<ResponseDto> responseEntity;
        try {
            final Pageable pageable = PageRequest.of(pageNumber, pageSize);
            final Page<Movie> allMovies = movieService.getAllMovies(pageable);
            responseEntity = createResponseDto(allMovies);
        } catch (final Exception e) {
            LOGGER.info("Exception occurred while fetching the response from the database.", e);
            return new ResponseEntity<>(HttpStatus.INTERNAL_SERVER_ERROR);
        }
        return responseEntity;
    }

    // @GetMapping annotation handles the http get request matched with the given uri.
    // @RequestParam annotation reads the request parameters (i.e. basically is a query string)
    // Sample urls -
    // 1. http://localhost:8089/api/movies/findByReleased
    // 2. http://localhost:8089/api/movies/findByReleased?isMovieReleased=false
    // 3. http://localhost:8089/api/movies/findByReleased?isMovieReleased=true&pageSize=15
    // 4. http://localhost:8089/api/movies/findByReleased?pageNumber=1&pageSize=15
    @GetMapping(value = "/findByReleased", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<ResponseDto> findByReleased(
            @RequestParam(name = "pageNumber", defaultValue = "0") final int pageNumber,    // In spring the default page number starts with '0'.
            @RequestParam(name = "pageSize", defaultValue = "5") final int pageSize,
            @RequestParam(name = "isMovieReleased", defaultValue = "true") final boolean isMovieReleased) {
        LOGGER.info("Getting all the movies from the database where movie released flag= {} for page-number= {} and " +
                "page-size= {}.", isMovieReleased, pageNumber, pageSize);
        final ResponseEntity<ResponseDto> responseEntity;
        try {
            final Pageable pageable = PageRequest.of(pageNumber, pageSize);
            final Page<Movie> allMovies = movieService.findByReleased(isMovieReleased, pageable);
            responseEntity = createResponseDto(allMovies);
        } catch (final Exception e) {
            LOGGER.info("Exception occurred while fetching the response from the database.", e);
            return new ResponseEntity<>(HttpStatus.INTERNAL_SERVER_ERROR);
        }
        return responseEntity;
    }

    private ResponseEntity<ResponseDto> createResponseDto(final Page<Movie> moviesPage) {
        final List<Movie> movies = moviesPage.getContent();
        final ResponseEntity<ResponseDto> responseEntity;
        if (CollectionUtils.isEmpty(movies)) {
            LOGGER.info("Returning an empty list as no movies are fetched from the database.");
            responseEntity = new ResponseEntity<>(ResponseDto.create(Collections.emptyList(), 0, 0, 0, null, null),
                    HttpStatus.OK);
        } else {
            responseEntity = new ResponseEntity<>(ResponseDto.create(movies, (int) moviesPage.getTotalElements(),
                    moviesPage.getTotalPages(), moviesPage.getNumber(), moviesPage.isFirst(), 
                    moviesPage.isLast()), HttpStatus.OK);
        }
        return responseEntity;
    }
}
