package com.springboot.methodlevel.security.controller;

import com.springboot.methodlevel.security.dto.UserDto;
import com.springboot.methodlevel.security.service.UserService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

//lombok annotation
@Slf4j
//spring annotations
@RestController
@RequestMapping("/api")
public class SecurityController {

    @Autowired
    UserService service;

    //note - @PreAuthorize checks for authorization before method execution

    //will be publicly accessible
    //URL - http://localhost:9800/api/anonymous
    @GetMapping("/anonymous")
    @ResponseStatus(HttpStatus.OK)
    public String getAnonymousResponse() {
        log.info("Returning anonymous response");
        return "Hello anonymous";
    }

    //will only be accessible by the user who has ROLE_USER assigned
    //URL - http://localhost:9800/api/protected/user
    @GetMapping("/protected/user")
    @ResponseStatus(HttpStatus.OK)
    @PreAuthorize("hasRole('USER')")
    public String getUserResponse() {
        log.info("Returning user response");
        return "Hello user";
    }

    //will be accessible by the users who has ROLE_MODERATOR assigned
    //URL - http://localhost:9800/api/protected/moderator
    @GetMapping("/protected/moderator")
    @ResponseStatus(HttpStatus.OK)
    @PreAuthorize("hasRole('MODERATOR')")
    public String getModeratorResponse() {
        log.info("Returning moderator response");
        return "Hello moderator";
    }

    //will be accessible by the users who has ROLE_ADMIN assigned
    //URL - http://localhost:9800/api/protected/admin
    @GetMapping("/protected/admin")
    @ResponseStatus(HttpStatus.OK)
    @PreAuthorize("hasRole('ADMIN')")
    public String getAdminResponse() {
        log.info("Returning administrator response");
        return "Hello administrator";
    }

    //will only be accessible by the user who has both ROLE_MODERATOR and ROLE_ADMIN assigned
    //URL - http://localhost:9800/api/protected/owner
    @GetMapping("/protected/owner")
    @ResponseStatus(HttpStatus.OK)
    @PreAuthorize("hasRole('MODERATOR') AND hasRole('ADMIN')")
    public String getAppOwnerResponse() {
        log.info("Returning application owner response response");
        return "Hello application owner";
    }

    //will only be accessible by the user who has both ROLE_MODERATOR and ROLE_ADMIN assigned
    //URL - http://localhost:9800/api/protected/get-all-users
    @GetMapping("/protected/get-all-users")
    @ResponseStatus(HttpStatus.OK)
    @PreAuthorize("hasRole('MODERATOR') AND hasRole('ADMIN')")
    public List<UserDto> getUsers() {
        log.info("Returning all users");
        return service.getUsers();
    }
}
