package com.jcg;

import com.jcg.model.Student;

import java.util.List;
import java.util.function.BiPredicate;

/*
Java8 functional interface - BiPredicate
represents a boolean valued function of two arguments and returns a boolean value
method syntax - <code>boolean test(T t, U u);</code>
 */
public class AppMain {
    // method #1
    public static void method1() {
        BiPredicate<String, Integer> filterByLength = (str1, length) -> str1.length() >= length;

        boolean isEqual = filterByLength.test("adam", 5);
        System.out.println(isEqual);

        boolean isEqual1 = filterByLength.test("geek", 4);
        System.out.println(isEqual1);
    }

    // method #2
    public static void method2() {
        List<Student> students = Student.create();

        BiPredicate<Student, String> filterByDepartment =
                (student, department) -> student.getDepartment().equals(department);

        for (Student student : students) {
            boolean result = filterByDepartment.test(student, "medical");
            if (result)
                System.out.println(student);
        }
    }

    // method #3
    // returns a composed predicate that represents the logical AND of one predicate and another
    // method syntax - <code>default BiPredicate and(BiPredicate other)</code>
    // similarly we can also have a composed predicate that represents the logical OR
    public static void method3() {
        List<Student> students = Student.create();

        BiPredicate<Student, String> namePrefixFilter = (student, prefix) -> student.getName().startsWith(prefix);

        BiPredicate<Student, String> nameSuffixFilter = (student, suffix) -> student.getName().endsWith(suffix);

        for (Student student : students) {
            boolean result = namePrefixFilter.and(nameSuffixFilter).test(student, "a");
            if (result)
                System.out.println(student);
        }
    }

    // method #4
    // returns a predicate that represents logical negation of the given predicate
    // method syntax - <code>default BiPredicate negate()</code>
    public static void method4() {
        BiPredicate<String, Integer> filterByLength = (string, length) -> string.length() >= length;

        // string length should not greater than or equal to 5
        System.out.println(filterByLength.negate().test("geeks", 5));
    }

    public static void main(String[] args) {
        System.out.println("-- BiPredicate functional interface implementation --\n");
        method1();
        System.out.println("\n");
        method2();
        System.out.println("\n");
        method3();
        System.out.println("\n");
        method4();
    }
}

