package org.jcg.producer;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.stereotype.Component;

import java.util.concurrent.CompletableFuture;

@Component
public class KafkaProducerV1 {

    @Autowired
    private KafkaTemplate<String, String> kafkaTemplate;

    @Value("${spring.kafka.topic-name}")
    private String topicName;

    private static final int MAX_RETRIES = 3;

    public CompletableFuture<Void> sendMessage(String message) {
        return sendWithRetryAndRecovery(message, 1);
    }

    private CompletableFuture<Void> sendWithRetryAndRecovery(String message, int retryCount) {
        return CompletableFuture.runAsync(() -> {
            kafkaTemplate.send(topicName, message);
            System.out.println("Message sent: " + message);
        }).handle((result, ex) -> {
            if (ex != null) {
                System.err.println("Error sending message: " + ex.getMessage());
                if (retryCount < MAX_RETRIES) {
                    System.out.println("Retrying message sending. Retry count: " + retryCount);
                    return sendWithRetryAndRecovery(message, retryCount + 1);
                } else {
                    System.err.println("Max retry attempts reached. Failed to send message.");
                    performRecoveryActions();
                }
            }
            return null; // Return null as the result since CompletableFuture<Void> doesn't expect a result
        });
    }

    private void performRecoveryActions() {
        // Implement your recovery logic here
        System.out.println("Performing recovery actions...");
        // Perform additional actions such as logging, fallback mechanisms, or notifying administrators
    }
}