/******************************************************************************
SFE_MicroOLED.cpp
Main source code for the MicroOLED Arduino Library

Jim Lindblom @ SparkFun Electronics
October 26, 2014
https://github.com/sparkfun/Micro_OLED_Breakout/tree/master/Firmware/Arduino/libraries/SFE_MicroOLED

Modified by:
Emil Varughese @ Edwin Robotics Pvt. Ltd.
July 27, 2015
https://github.com/emil01/SparkFun_Micro_OLED_Arduino_Library/

This file defines the hardware interface(s) for the Micro OLED Breakout. Those
interfaces include SPI, I2C and a parallel bus.

Development environment specifics:
Arduino 1.0.5
Arduino Pro 3.3V
Micro OLED Breakout v1.0

This code was heavily based around the MicroView library, written by GeekAmmo
(https://github.com/geekammo/MicroView-Arduino-Library), and released under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
******************************************************************************/
#include <Arduino.h>
#if defined(ARDUINO_ARCH_MBED)
	// ARDUINO_ARCH_MBED (APOLLO3 v2) does not support or require pgmspace.h / PROGMEM
#elif defined(__AVR__) || defined(__arm__) || defined(__ARDUINO_ARC__)
	#include <avr/pgmspace.h>
#else
	#include <pgmspace.h>
#endif
#include <SFE_MicroOLED.h>

#ifndef _BV
#define _BV(x) (1 << x)
#endif

// This fixed ugly GCC warning "only initialized variables can be placed into program memory area"
#if defined(__AVR__)
#undef PROGMEM
#define PROGMEM __attribute__((section(".progmem.data")))
#endif

// Add header of the fonts here.
// Fonts that aren't included the section below are excluded by the compiler.
#include "util/font5x7.h"				// Font 0
#include "util/font8x16.h"				// Font 1
#include "util/7segment.h"				// Font 2
#include "util/fontlargenumber.h"		// Font 3
#include "util/fontlargeletter31x48.h"	// Font 4 (excluded by default - see below)

#define MAXFONTS 5 // Do not change this line - except when _adding_ new fonts

// To save flash memory, change these to zeros for the fonts you want to exclude.
// In particular, the 31x48 font is handy, but uses a big
// chunk of flash memory - about 7k. It is excluded by default.
//
// If you are compiling the code using your own makefile, you can use compiler flags to include 
// or exclude individual fonts. E.g.:  -DINCLUDE_FONT_LARGELETTER=1  or  -DINCLUDE_FONT_LARGENUMBER=0
#ifndef INCLUDE_FONT_5x7
#define INCLUDE_FONT_5x7 1			// Change this to 0 to exclude the 5x7 font
#endif
#ifndef INCLUDE_FONT_8x16
#define INCLUDE_FONT_8x16 1			// Change this to 0 to exclude the 8x16 font
#endif
#ifndef INCLUDE_FONT_7SEG
#define INCLUDE_FONT_7SEG 1			// Change this to 0 to exclude the seven segment font
#endif
#ifndef INCLUDE_FONT_LARGENUMBER
#define INCLUDE_FONT_LARGENUMBER 1	// Change this to 0 to exclude the large number font
#endif
#ifndef INCLUDE_FONT_LARGELETTER
#define INCLUDE_FONT_LARGELETTER 0	// Change this to 1 to include the large letter font
#endif


// Add the font name as declared in the header file.
// Exclude as many as possible to conserve FLASH memory.
const unsigned char *MicroOLED::fontsPointer[] = {
#if INCLUDE_FONT_5x7
	font5x7,
#else
    NULL,
#endif
#if INCLUDE_FONT_8x16
	font8x16,
#else
    NULL,
#endif
#if INCLUDE_FONT_7SEG
	sevensegment,
#else
    NULL,
#endif
#if INCLUDE_FONT_LARGENUMBER
    fontlargenumber,
#else
    NULL,
#endif
#if INCLUDE_FONT_LARGELETTER
	fontlargeletter31x48
#else
    NULL
#endif
};

/** \brief MicroOLED screen buffer.

Page buffer 64 x 48 divided by 8 = 384 bytes
Page buffer is required because in SPI mode, the host cannot read the SSD1306's GDRAM of the controller.  This page buffer serves as a scratch RAM for graphical functions.  All drawing function will first be drawn on this page buffer, only upon calling display() function will transfer the page buffer to the actual LCD controller's memory.
*/
static uint8_t screenmemory[] = {
	/* LCD Memory organised in 64 horizontal pixel and 6 rows of byte
	 B  B .............B  -----
	 y  y .............y        \
	 t  t .............t         \
	 e  e .............e          \
	 0  1 .............63          \
	                                \
	 D0 D0.............D0            \
	 D1 D1.............D1            / ROW 0
	 D2 D2.............D2           /
	 D3 D3.............D3          /
	 D4 D4.............D4         /
	 D5 D5.............D5        /
	 D6 D6.............D6       /
	 D7 D7.............D7  ----
	*/
	//SparkFun Electronics LOGO

	// ROW0, BYTE0 to BYTE63
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xF8, 0xFC, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0x0F, 0x07, 0x07, 0x06, 0x06, 0x00, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	// ROW1, BYTE64 to BYTE127
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x81, 0x07, 0x0F, 0x3F, 0x3F, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFE, 0xFE, 0xFC, 0xFC, 0xFC, 0xFE, 0xFF, 0xFF, 0xFF, 0xFC, 0xF8, 0xE0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	// ROW2, BYTE128 to BYTE191
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFC,
	0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF1, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xF0, 0xFD, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	// ROW3, BYTE192 to BYTE255
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0x3F, 0x1F, 0x07, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	// ROW4, BYTE256 to BYTE319
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0x3F, 0x1F, 0x1F, 0x0F, 0x0F, 0x0F, 0x0F,
	0x0F, 0x0F, 0x0F, 0x0F, 0x07, 0x07, 0x07, 0x03, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	// ROW5, BYTE320 to BYTE383
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,
	0x7F, 0x3F, 0x1F, 0x0F, 0x07, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

/** \brief MicroOLED Constructor -- I2C - leaving the address currently undefined

	Setup the MicroOLED class, assuming that the I2C address will be defined later.
*/
MicroOLED::MicroOLED(uint8_t rst)
{
	// Assign each of the parameters to a private class variable.
	rstPin = rst;
	moled_interface = MOLED_MODE_I2C; // Set interface to I2C
	moled_i2c_address = I2C_ADDRESS_UNDEFINED; // Flag that the I2C address is undefined
}

/** \brief MicroOLED Constructor -- SPI Mode

	Setup the MicroOLED class, configure the display to be controlled via a
	SPI interface.
*/
MicroOLED::MicroOLED(uint8_t rst, uint8_t dc, uint8_t cs)
{
	// Assign each of the parameters to a private class variable.
	rstPin = rst;
	dcPin = dc;
	csPin = cs;
	moled_interface = MOLED_MODE_SPI; // Set interface mode to SPI
	//_spiPort will be initialized by spiSetup
}

/** \brief MicroOLED Constructor -- I2C Mode

	Setup the MicroOLED class, configure the display to be controlled via a
	I2C interface.
*/
MicroOLED::MicroOLED(uint8_t rst, uint8_t dc)
{
	rstPin = rst;		  // Assign reset pin to private class variable
	moled_interface = MOLED_MODE_I2C; // Set interface to I2C
	// Set the I2C Address based on whether DC is high (1) or low (0).
	// The pin is pulled low by default, so if it's not explicitly set to
	// 1, just default to 0.
	if (dc == 1)
		moled_i2c_address = I2C_ADDRESS_SA0_1;
	else
		moled_i2c_address = I2C_ADDRESS_SA0_0;
	//_i2cPort will be initialized by i2cSetup
}

/** \brief MicroOLED Constructor -- Parallel Mode

	Setup the MicroOLED class, configure the display to be controlled via a
	parallel interface.
*/
MicroOLED::MicroOLED(uint8_t rst, uint8_t dc, uint8_t cs, uint8_t wr, uint8_t rd,
					 uint8_t d0, uint8_t d1, uint8_t d2, uint8_t d3,
					 uint8_t d4, uint8_t d5, uint8_t d6, uint8_t d7)
{
	moled_interface = MOLED_MODE_PARALLEL; // Set to parallel mode
	// Assign pin parameters to private class variables.
	rstPin = rst;
	dcPin = dc;
	csPin = cs;
	wrPin = wr;
	rdPin = rd;
	dPins[0] = d0;
	dPins[1] = d1;
	dPins[2] = d2;
	dPins[3] = d3;
	dPins[4] = d4;
	dPins[5] = d5;
	dPins[6] = d6;
	dPins[7] = d7;
}

/** \brief Initialisation of MicroOLED Library.

    Setup for the chosen interface then send initialisation commands to the SSD1306 controller inside the OLED.
*/
boolean MicroOLED::begin()
{
	// Set up the selected interface:
	if (moled_interface == MOLED_MODE_SPI)
		spiSetup();
	else if (moled_interface == MOLED_MODE_I2C)
		i2cSetup();
	else if (moled_interface == MOLED_MODE_PARALLEL)
		parallelSetup();
	else //if (moled_interface == MOLED_MODE_UNDEFINED)
		return (false);

	// Trap if the user instantiated with MicroOLED oled(PIN_RESET) and then called
	// .begin instead of .begin(uint8_t deviceAddress, TwoWire &wirePort)
	if ((moled_interface == MOLED_MODE_I2C) && (moled_i2c_address == I2C_ADDRESS_UNDEFINED))
	{
		if (_printDebug == true)
		{
			_debugPort->println(F("begin: error! deviceAddress is I2C_ADDRESS_UNDEFINED!"));
			_debugPort->println(F("begin: Did you forget to call .begin(uint8_t deviceAddress, TwoWire &wirePort)?"));
		}
		return (false);
	}

	beginCommon();
	return (true);
}

/** \brief Initialisation of MicroOLED Library.

    Setup IO pins for the SPI interface then send initialisation commands to the SSD1306 controller inside the OLED.
*/
boolean MicroOLED::begin(SPIClass &spiPort)
{
	// Set up the selected interface:
	spiSetup(spiPort);

	beginCommon();
	return (true);
}

/** \brief Initialisation of MicroOLED Library.

    Setup IO pins for the I2C interface then send initialisation commands to the SSD1306 controller inside the OLED.
*/
boolean MicroOLED::begin(uint8_t deviceAddress, TwoWire &wirePort)
{
	// Set up the selected interface:
	i2cSetup(deviceAddress, wirePort);

	beginCommon();
	return (true);
}

/** \brief Initialisation of MicroOLED Library - common to all begin methods.

    Setup IO pins for the chosen interface then send initialisation commands to the SSD1306 controller inside the OLED.
*/
void MicroOLED::beginCommon()
{
	// default 5x7 font
	setFontType(0);
	setColor(WHITE);
	setDrawMode(NORM);
	setCursor(0, 0);

	if(rstPin != 255)
	{
		// Display reset routine
		pinMode(rstPin, OUTPUT);	// Set RST pin as OUTPUT
		digitalWrite(rstPin, HIGH); // Initially set RST HIGH
		delay(5);					// VDD (3.3V) goes high at start, lets just chill for 5 ms
		digitalWrite(rstPin, LOW);	// Bring RST low, reset the display
		delay(10);					// wait 10ms
		digitalWrite(rstPin, HIGH); // Set RST HIGH, bring out of reset
	}

	// Display Init sequence for 64x48 OLED module
	command(DISPLAYOFF); // 0xAE

	initDisplay();

	command(DISPLAYON); //--turn on oled panel
}

/** \brief Set CGRAM and display settings

    Set the unique SSD1306 settings for the MicroOLED setup.
*/
void MicroOLED::initDisplay(bool clearDisplay)
{
	command(SETDISPLAYCLOCKDIV); // 0xD5
	command(0x80);				 // the suggested ratio 0x80

	command(SETMULTIPLEX); // 0xA8
	command(0x2F);

	command(SETDISPLAYOFFSET); // 0xD3
	command(0x0);			   // no offset

	command(SETSTARTLINE | 0x0); // line #0

	command(CHARGEPUMP); // enable charge pump
	command(0x14);

	command(NORMALDISPLAY);		 // 0xA6
	command(DISPLAYALLONRESUME); // 0xA4

	command(SEGREMAP | 0x1);
	command(COMSCANDEC);

	command(SETCOMPINS); // 0xDA
	command(0x12);

	command(SETCONTRAST); // 0x81
	command(0x8F);

	command(SETPRECHARGE); // 0xd9
	command(0xF1);

	command(SETVCOMDESELECT); // 0xDB
	command(0x40);

	if(clearDisplay)
		clear(ALL);			// Erase hardware memory inside the OLED controller to avoid random data in memory.
}

//Calling this function with nothing sets the debug port to Serial
//You can also call it with other streams like Serial1, SerialUSB, etc.
void MicroOLED::enableDebugging(Stream &debugPort)
{
	_debugPort = &debugPort;
	_printDebug = true;
}

/** \brief Send the display a command byte

    Send a command via SPI, I2C or parallel	to SSD1306 controller.
	For SPI we set the DC and CS pins here, and call spiTransfer(byte)
	to send the data. For I2C and Parallel we use the write functions
	defined in hardware.cpp to send the data.
*/
void MicroOLED::command(uint8_t c)
{

	if (moled_interface == MOLED_MODE_SPI)
	{
		digitalWrite(dcPin, LOW);
		;				// DC pin LOW for a command
		spiTransfer(c); // Transfer the command byte
	}
	else if (moled_interface == MOLED_MODE_I2C)
	{
		// Write to our address, make sure it knows we're sending a
		// command:
		i2cWrite(moled_i2c_address, I2C_COMMAND, c);
	}
	else if (moled_interface == MOLED_MODE_PARALLEL)
	{
		// Write the byte to our parallel interface. Set DC LOW.
		parallelWrite(c, LOW);
	}
}

/** \brief Send the display a data byte

    Send a data byte via SPI, I2C or parallel to SSD1306 controller.
	For SPI we set the DC and CS pins here, and call spiTransfer(byte)
	to send the data. For I2C and Parallel we use the write functions
	defined in hardware.cpp to send the data.
*/
void MicroOLED::data(uint8_t c)
{

	if (moled_interface == MOLED_MODE_SPI)
	{
		digitalWrite(dcPin, HIGH); // DC HIGH for a data byte

		spiTransfer(c); // Transfer the data byte
	}
	else if (moled_interface == MOLED_MODE_I2C)
	{
		// Write to our address, make sure it knows we're sending a
		// data byte:
		i2cWrite(moled_i2c_address, I2C_DATA, c);
	}
	else if (moled_interface == MOLED_MODE_PARALLEL)
	{
		// Write the byte to our parallel interface. Set DC HIGH.
		parallelWrite(c, HIGH);
	}
}

/** \brief Set SSD1306 page address.

    Send page address command and address to the SSD1306 OLED controller.
*/
void MicroOLED::setPageAddress(uint8_t add)
{
	add = 0xb0 | add;
	command(add);
	return;
}

/** \brief Set SSD1306 column address.

    Send column address command and address to the SSD1306 OLED controller.
*/
void MicroOLED::setColumnAddress(uint8_t add)
{
	command((0x10 | (add >> 4)) + 0x02);
	command((0x0f & add));
	return;
}

/** \brief Clear screen buffer or SSD1306's memory.

    To clear GDRAM inside the LCD controller, pass in the variable mode = ALL and to clear screen page buffer pass in the variable mode = PAGE.
*/
void MicroOLED::clear(uint8_t mode)
{
	//	uint8_t page=6, col=0x40;
	if (mode == ALL)
	{
		for (int i = 0; i < 8; i++)
		{
			setPageAddress(i);
			setColumnAddress(0);
			if (moled_interface == MOLED_MODE_I2C)
			{
				uint8_t zeros[0x80];
				memset(zeros, 0, 0x80);
				i2cWriteMultiple(moled_i2c_address, (uint8_t *)&zeros, 0x80);
			}
			else
			{
				for (int j = 0; j < 0x80; j++)
				{
					data(0);
				}
			}
		}
	}
	else
	{
		memset(screenmemory, 0, 384); // (64 x 48) / 8 = 384
									  //display();
	}
}

/** \brief Clear or replace screen buffer or SSD1306's memory with a character.

	To clear GDRAM inside the LCD controller, pass in the variable mode = ALL with c character and to clear screen page buffer, pass in the variable mode = PAGE with c character.
*/
void MicroOLED::clear(uint8_t mode, uint8_t c)
{
	//uint8_t page=6, col=0x40;
	if (mode == ALL)
	{
		for (int i = 0; i < 8; i++)
		{
			setPageAddress(i);
			setColumnAddress(0);
			if (moled_interface == MOLED_MODE_I2C)
			{
				uint8_t zeros[0x80];
				memset(zeros, c, 0x80);
				i2cWriteMultiple(moled_i2c_address, (uint8_t *)&zeros, 0x80);
			}
			else
			{
				for (int j = 0; j < 0x80; j++)
				{
					data(c);
				}
			}
		}
	}
	else
	{
		memset(screenmemory, c, 384); // (64 x 48) / 8 = 384
		display();
	}
}

/** \brief Invert display.

    The WHITE color of the display will turn to BLACK and the BLACK will turn to WHITE.
*/
void MicroOLED::invert(boolean inv)
{
	if (inv)
		command(INVERTDISPLAY);
	else
		command(NORMALDISPLAY);
}

/** \brief Set contrast.

    OLED contract value from 0 to 255. Note: Contrast level is not very obvious.
*/
void MicroOLED::contrast(uint8_t contrast)
{
	command(SETCONTRAST); // 0x81
	command(contrast);
}

/** \brief Transfer display memory.

    Bulk move the screen buffer to the SSD1306 controller's memory so that images/graphics drawn on the screen buffer will be displayed on the OLED.
*/
void MicroOLED::display(void)
{
	uint8_t i, j;

	for (i = 0; i < 6; i++)
	{
		setPageAddress(i);
		setColumnAddress(0);
		if (moled_interface == MOLED_MODE_I2C)
		{
			i2cWriteMultiple(moled_i2c_address, (uint8_t *)&screenmemory[i * 0x40], 0x40);
		}
		else
		{
			for (j = 0; j < 0x40; j++)
			{
				data(screenmemory[i * 0x40 + j]);
			}
		}
	}
}

/** \brief Override Arduino's Print.

    Arduino's print overridden so that we can use uView.print().
*/
size_t MicroOLED::write(uint8_t c)
{
	if (c == '\n')
	{
		cursorY += fontHeight;
		cursorX = 0;
	}
	else if (c == '\r')
	{
		// skip
	}
	else
	{
		drawChar(cursorX, cursorY, c, foreColor, drawMode);
		cursorX += fontWidth + 1;
		if ((cursorX > (LCDWIDTH - fontWidth)))
		{
			cursorY += fontHeight;
			cursorX = 0;
		}
	}

	return 1;
}

/** \brief Set cursor position.

MicroOLED's cursor position to x,y.
*/
void MicroOLED::setCursor(uint8_t x, uint8_t y)
{
	cursorX = x;
	cursorY = y;
}

/** \brief Draw pixel.

Draw pixel using the current fore color and current draw mode in the screen buffer's x,y position.
*/
void MicroOLED::pixel(uint8_t x, uint8_t y)
{
	pixel(x, y, foreColor, drawMode);
}

/** \brief Draw pixel with color and mode.

Draw color pixel in the screen buffer's x,y position with NORM or XOR draw mode.
*/
void MicroOLED::pixel(uint8_t x, uint8_t y, uint8_t color, uint8_t mode)
{
	if ((x >= LCDWIDTH) || (y >= LCDHEIGHT))
		return;

	if (mode == XOR)
	{
		if (color == WHITE)
			screenmemory[x + (y / 8) * LCDWIDTH] ^= _BV((y % 8));
	}
	else
	{
		if (color == WHITE)
			screenmemory[x + (y / 8) * LCDWIDTH] |= _BV((y % 8));
		else
			screenmemory[x + (y / 8) * LCDWIDTH] &= ~_BV((y % 8));
	}
}

/** \brief Draw line.

Draw line using current fore color and current draw mode from x0,y0 to x1,y1 of the screen buffer.
*/
void MicroOLED::line(uint8_t x0, uint8_t y0, uint8_t x1, uint8_t y1)
{
	line(x0, y0, x1, y1, foreColor, drawMode);
}

/** \brief Draw line with color and mode.

Draw line using color and mode from x0,y0 to x1,y1 of the screen buffer.
*/
void MicroOLED::line(uint8_t x0, uint8_t y0, uint8_t x1, uint8_t y1, uint8_t color, uint8_t mode)
{
	if (_printDebug == true)
	{
		_debugPort->print(F("line: line coords: ("));
		_debugPort->print(x0);
		_debugPort->print(F(","));
		_debugPort->print(y0);
		_debugPort->print(F(") ("));
		_debugPort->print(x1);
		_debugPort->print(F(","));
		_debugPort->print(y1);
		_debugPort->println(F(")"));
	}

	uint8_t steep = abs(y1 - y0) > abs(x1 - x0);
	if (steep)
	{
		swapOLED(&x0, &y0);
		swapOLED(&x1, &y1);
		if (_printDebug == true)
			_debugPort->println(F("line: line is steep"));
}

	if (x0 > x1)
	{
		swapOLED(&x0, &x1);
		swapOLED(&y0, &y1);
		if (_printDebug == true)
			_debugPort->println(F("line: x0 > x1"));
	}

	// if (_printDebug == true)
	// {
	// 	_debugPort->print(F("line: line coords: ("));
	// 	_debugPort->print(x0);
	// 	_debugPort->print(F(","));
	// 	_debugPort->print(y0);
	// 	_debugPort->print(F(") ("));
	// 	_debugPort->print(x1);
	// 	_debugPort->print(F(","));
	// 	_debugPort->print(y1);
	// 	_debugPort->println(F(")"));
	// }

	uint8_t dx, dy;
	dx = x1 - x0;
	dy = abs(y1 - y0);

	if (_printDebug == true)
	{
		_debugPort->print(F("line: dx: "));
		_debugPort->print(dx);
		_debugPort->print(F("  dy: "));
		_debugPort->println(dy);
	}

	if ((dx == 0) && (dy == 0))
	{
		if (_printDebug == true)
			_debugPort->print(F("line: zero length!"));
		pixel(x0, y0, color, mode);
		return;
	}

	int8_t err = dx / 2;
	int8_t ystep;

	if (_printDebug == true)
	{
		_debugPort->print(F("line: err: "));
		_debugPort->println(err);
	}

	if (y0 < y1)
	{
		ystep = 1;
	}
	else
	{
		ystep = -1;
	}

	if (_printDebug == true)
	{
		_debugPort->print(F("line: ystep: "));
		_debugPort->println(ystep);
	}

	for (; x0 <= x1; x0++)
	{
		if (steep)
		{
			pixel(y0, x0, color, mode);
			if (_printDebug == true)
			{
				_debugPort->print(F("line: steep pixel: ("));
				_debugPort->print(y0);
				_debugPort->print(F(","));
				_debugPort->print(x0);
				_debugPort->println(F(")"));
			}
		}
		else
		{
			pixel(x0, y0, color, mode);
			if (_printDebug == true)
			{
				_debugPort->print(F("line: pixel: ("));
				_debugPort->print(x0);
				_debugPort->print(F(","));
				_debugPort->print(y0);
				_debugPort->println(F(")"));
			}
		}
		err -= dy;
		if (err < 0)
		{
			y0 += ystep;
			err += dx;
		}
	}
}

/** \brief Draw horizontal line.

Draw horizontal line using current fore color and current draw mode from x,y to x+width,y of the screen buffer.
*/
void MicroOLED::lineH(uint8_t x, uint8_t y, uint8_t width)
{
	line(x, y, x + width, y, foreColor, drawMode);
}

/** \brief Draw horizontal line with color and mode.

Draw horizontal line using color and mode from x,y to x+width,y of the screen buffer.
*/
void MicroOLED::lineH(uint8_t x, uint8_t y, uint8_t width, uint8_t color, uint8_t mode)
{
	line(x, y, x + width, y, color, mode);
}

/** \brief Draw vertical line.

Draw vertical line using current fore color and current draw mode from x,y to x,y+height of the screen buffer.
*/
void MicroOLED::lineV(uint8_t x, uint8_t y, uint8_t height)
{
	line(x, y, x, y + height, foreColor, drawMode);
}

/** \brief Draw vertical line with color and mode.

Draw vertical line using color and mode from x,y to x,y+height of the screen buffer.
*/
void MicroOLED::lineV(uint8_t x, uint8_t y, uint8_t height, uint8_t color, uint8_t mode)
{
	line(x, y, x, y + height, color, mode);
}

/** \brief Draw rectangle.

Draw rectangle using current fore color and current draw mode from x,y to x+width,y+height of the screen buffer.
*/
void MicroOLED::rect(uint8_t x, uint8_t y, uint8_t width, uint8_t height)
{
	rect(x, y, width, height, foreColor, drawMode);
}

/** \brief Draw rectangle with color and mode.

Draw rectangle using color and mode from x,y to x+width,y+height of the screen buffer.
*/
void MicroOLED::rect(uint8_t x, uint8_t y, uint8_t width, uint8_t height, uint8_t color, uint8_t mode)
{
	uint8_t tempHeight;

	lineH(x, y, width, color, mode);
	lineH(x, y + height - 1, width, color, mode);

	tempHeight = height - 2;

	// skip drawing vertical lines to avoid overlapping of pixel that will
	// affect XOR plot if no pixel in between horizontal lines
	if (tempHeight < 1)
		return;

	lineV(x, y + 1, tempHeight, color, mode);
	lineV(x + width - 1, y + 1, tempHeight, color, mode);
}

/** \brief Draw filled rectangle.

Draw filled rectangle using current fore color and current draw mode from x,y to x+width,y+height of the screen buffer.
*/
void MicroOLED::rectFill(uint8_t x, uint8_t y, uint8_t width, uint8_t height)
{
	rectFill(x, y, width, height, foreColor, drawMode);
}

/** \brief Draw filled rectangle with color and mode.

Draw filled rectangle using color and mode from x,y to x+width,y+height of the screen buffer.
*/
void MicroOLED::rectFill(uint8_t x, uint8_t y, uint8_t width, uint8_t height, uint8_t color, uint8_t mode)
{
	// TODO - need to optimise the memory map draw so that this function will not call pixel one by one
	for (int i = x; i < x + width; i++)
	{
		lineV(i, y, height, color, mode);
	}
}

/** \brief Draw circle.

    Draw circle with radius using current fore color and current draw mode at x,y of the screen buffer.
*/
void MicroOLED::circle(uint8_t x0, uint8_t y0, uint8_t radius)
{
	circle(x0, y0, radius, foreColor, drawMode);
}

/** \brief Draw circle with color and mode.

Draw circle with radius using color and mode at x,y of the screen buffer.
*/
void MicroOLED::circle(uint8_t x0, uint8_t y0, uint8_t radius, uint8_t color, uint8_t mode)
{
	//TODO - find a way to check for no overlapping of pixels so that XOR draw mode will work perfectly
	int8_t f = 1 - radius;
	int8_t ddF_x = 1;
	int8_t ddF_y = -2 * radius;
	int8_t x = 0;
	int8_t y = radius;

	pixel(x0, y0 + radius, color, mode);
	pixel(x0, y0 - radius, color, mode);
	pixel(x0 + radius, y0, color, mode);
	pixel(x0 - radius, y0, color, mode);

	while (x < y)
	{
		if (f >= 0)
		{
			y--;
			ddF_y += 2;
			f += ddF_y;
		}
		x++;
		ddF_x += 2;
		f += ddF_x;

		pixel(x0 + x, y0 + y, color, mode);
		pixel(x0 - x, y0 + y, color, mode);
		pixel(x0 + x, y0 - y, color, mode);
		pixel(x0 - x, y0 - y, color, mode);

		pixel(x0 + y, y0 + x, color, mode);
		pixel(x0 - y, y0 + x, color, mode);
		pixel(x0 + y, y0 - x, color, mode);
		pixel(x0 - y, y0 - x, color, mode);
	}
}

/** \brief Draw filled circle.

    Draw filled circle with radius using current fore color and current draw mode at x,y of the screen buffer.
*/
void MicroOLED::circleFill(uint8_t x0, uint8_t y0, uint8_t radius)
{
	circleFill(x0, y0, radius, foreColor, drawMode);
}

/** \brief Draw filled circle with color and mode.

    Draw filled circle with radius using color and mode at x,y of the screen buffer.
*/
void MicroOLED::circleFill(uint8_t x0, uint8_t y0, uint8_t radius, uint8_t color, uint8_t mode)
{
	// TODO - - find a way to check for no overlapping of pixels so that XOR draw mode will work perfectly
	int8_t f = 1 - radius;
	int8_t ddF_x = 1;
	int8_t ddF_y = -2 * radius;
	int8_t x = 0;
	int8_t y = radius;

	// Temporary disable fill circle for XOR mode.
	if (mode == XOR)
		return;

	for (uint8_t i = y0 - radius; i <= y0 + radius; i++)
	{
		pixel(x0, i, color, mode);
	}

	while (x < y)
	{
		if (f >= 0)
		{
			y--;
			ddF_y += 2;
			f += ddF_y;
		}
		x++;
		ddF_x += 2;
		f += ddF_x;

		for (uint8_t i = y0 - y; i <= y0 + y; i++)
		{
			pixel(x0 + x, i, color, mode);
			pixel(x0 - x, i, color, mode);
		}
		for (uint8_t i = y0 - x; i <= y0 + x; i++)
		{
			pixel(x0 + y, i, color, mode);
			pixel(x0 - y, i, color, mode);
		}
	}
}

/** \brief Get LCD height.

    The height of the LCD return as byte.
*/
uint8_t MicroOLED::getLCDHeight(void)
{
	return LCDHEIGHT;
}

/** \brief Get LCD width.

    The width of the LCD return as byte.
*/
uint8_t MicroOLED::getLCDWidth(void)
{
	return LCDWIDTH;
}

/** \brief Get font width.

    The cucrrent font's width return as byte.
*/
uint8_t MicroOLED::getFontWidth(void)
{
	return fontWidth;
}

/** \brief Get font height.

    The current font's height return as byte.
*/
uint8_t MicroOLED::getFontHeight(void)
{
	return fontHeight;
}

/** \brief Get font starting character.

    Return the starting ASCII character of the currnet font, not all fonts start with ASCII character 0. Custom fonts can start from any ASCII character.
*/
uint8_t MicroOLED::getFontStartChar(void)
{
	return fontStartChar;
}

/** \brief Get font total characters.

    Return the total characters of the current font.
*/
uint8_t MicroOLED::getFontTotalChar(void)
{
	return fontTotalChar;
}

/** \brief Get total fonts.

    Return the total number of fonts loaded into the MicroOLED's flash memory.
*/
uint8_t MicroOLED::getTotalFonts(void)
{
	uint8_t totalFonts = 0;
	for (uint8_t thisFont = 0; thisFont < MAXFONTS; thisFont++)
	{
		if (fontsPointer[thisFont] != NULL)
			totalFonts++;
	}
	return (totalFonts);
}

/** \brief Get font type.

    Return the font type number of the current font.
*/
uint8_t MicroOLED::getFontType(void)
{
	return fontType;
}

/** \brief Set font type.

    Set the current font type number, ie changing to different fonts base on the type provided.
*/
uint8_t MicroOLED::setFontType(uint8_t type)
{
    if (type >= MAXFONTS)
		return false;
	if (fontsPointer[type] == NULL)
        return false;

	fontType = type;
	fontWidth = pgm_read_byte(fontsPointer[fontType] + 0);
	fontHeight = pgm_read_byte(fontsPointer[fontType] + 1);
	fontStartChar = pgm_read_byte(fontsPointer[fontType] + 2);
	fontTotalChar = pgm_read_byte(fontsPointer[fontType] + 3);
	fontMapWidth = (pgm_read_byte(fontsPointer[fontType] + 4) * 100) + pgm_read_byte(fontsPointer[fontType] + 5); // two bytes values into integer 16
	return true;
}

/** \brief Set color.

    Set the current draw's color. Only WHITE and BLACK available.
*/
void MicroOLED::setColor(uint8_t color)
{
	foreColor = color;
}

/** \brief Set draw mode.

    Set current draw mode with NORM or XOR.
*/
void MicroOLED::setDrawMode(uint8_t mode)
{
	drawMode = mode;
}

/** \brief Draw character.

    Draw character c using current color and current draw mode at x,y.
*/
void MicroOLED::drawChar(uint8_t x, uint8_t y, uint8_t c)
{
	drawChar(x, y, c, foreColor, drawMode);
}

/** \brief Draw character with color and mode.

    Draw character c using color and draw mode at x,y.
*/
void MicroOLED::drawChar(uint8_t x, uint8_t y, uint8_t c, uint8_t color, uint8_t mode)
{
	// TODO - New routine to take font of any height, at the moment limited to font height in multiple of 8 pixels

	uint8_t rowsToDraw, row, tempC;
	uint8_t i, j, temp;
	uint16_t charPerBitmapRow, charColPositionOnBitmap, charRowPositionOnBitmap, charBitmapStartPosition;

	if ((c < fontStartChar) || (c > (fontStartChar + fontTotalChar - 1))) // no bitmap for the required c
		return;

	tempC = c - fontStartChar;

	// each row (in datasheet is call page) is 8 bits high, 16 bit high character will have 2 rows to be drawn
	rowsToDraw = fontHeight / 8; // 8 is LCD's page size, see SSD1306 datasheet
	if (rowsToDraw <= 1)
		rowsToDraw = 1;

	// the following draw function can draw anywhere on the screen, but SLOW pixel by pixel draw
	if (rowsToDraw == 1)
	{
		for (i = 0; i < fontWidth + 1; i++)
		{
			if (i == fontWidth) // this is done in a weird way because for 5x7 font, there is no margin, this code add a margin after col 5
				temp = 0;
			else
				temp = pgm_read_byte(fontsPointer[fontType] + FONTHEADERSIZE + (tempC * fontWidth) + i);

			for (j = 0; j < 8; j++)
			{ // 8 is the LCD's page height (see datasheet for explanation)
				if (temp & 0x1)
				{
					pixel(x + i, y + j, color, mode);
				}
				else
				{
					pixel(x + i, y + j, !color, mode);
				}

				temp >>= 1;
			}
		}
		return;
	}

	// font height over 8 bit
	// take character "0" ASCII 48 as example
	charPerBitmapRow = fontMapWidth / fontWidth;			 // 256/8 =32 char per row
	charColPositionOnBitmap = tempC % charPerBitmapRow;		 // =16
	charRowPositionOnBitmap = int(tempC / charPerBitmapRow); // =1
	charBitmapStartPosition = (charRowPositionOnBitmap * fontMapWidth * (fontHeight / 8)) + (charColPositionOnBitmap * fontWidth);

	// each row on LCD is 8 bit height (see datasheet for explanation)
	for (row = 0; row < rowsToDraw; row++)
	{
		for (i = 0; i < fontWidth; i++)
		{
			temp = pgm_read_byte(fontsPointer[fontType] + FONTHEADERSIZE + (charBitmapStartPosition + i + (row * fontMapWidth)));
			for (j = 0; j < 8; j++)
			{ // 8 is the LCD's page height (see datasheet for explanation)
				if (temp & 0x1)
				{
					pixel(x + i, y + j + (row * 8), color, mode);
				}
				else
				{
					pixel(x + i, y + j + (row * 8), !color, mode);
				}
				temp >>= 1;
			}
		}
	}
}

/** \brief Stop scrolling.

    Stop the scrolling of graphics on the OLED.
*/
void MicroOLED::scrollStop(void)
{
	command(DEACTIVATESCROLL);
}

/** \brief Right scrolling.

Set row start to row stop on the OLED to scroll right.
*/
void MicroOLED::scrollRight(uint8_t start, uint8_t stop, uint8_t scrollInterval)
{
	if (stop < start) // stop must be larger or equal to start
		return;
	scrollStop(); // need to disable scrolling before starting to avoid memory corrupt
	command(RIGHTHORIZONTALSCROLL);
	command(0x00);
	command(start);
	command(scrollInterval);
	command(stop);
	command(0x00);
	command(0xFF);
	command(ACTIVATESCROLL);
}

/** \brief Vert Right scrolling.

Set row start to row stop on the OLED to scroll vert right.
*/
void MicroOLED::scrollVertRight(uint8_t start, uint8_t stop, uint8_t scrollInterval)
{
	if (stop < start) // stop must be larger or equal to start
		return;
	scrollStop(); // need to disable scrolling before starting to avoid memory corrupt
	command(VERTICALRIGHTHORIZONTALSCROLL);
	command(0x00);
	command(start);
	command(scrollInterval);
	command(stop);
	command(0x01); // Vertical scrolling offset
	command(ACTIVATESCROLL);
}

/** \brief Left scrolling.

Set row start to row stop on the OLED to scroll left.
*/
void MicroOLED::scrollLeft(uint8_t start, uint8_t stop, uint8_t scrollInterval)
{
	if (stop < start) // stop must be larger or equal to start
		return;
	scrollStop(); // need to disable scrolling before starting to avoid memory corrupt
	command(LEFT_HORIZONTALSCROLL);
	command(0x00);
	command(start);
	command(scrollInterval);
	command(stop);
	command(0x00);
	command(0xFF);
	command(ACTIVATESCROLL);
}

/** \brief Vert Left scrolling.

Set row start to row stop on the OLED to scroll vert left.
*/
void MicroOLED::scrollVertLeft(uint8_t start, uint8_t stop, uint8_t scrollInterval)
{
	if (stop < start) // stop must be larger or equal to start
		return;
	scrollStop(); // need to disable scrolling before starting to avoid memory corrupt
	command(VERTICALLEFTHORIZONTALSCROLL);
	command(0x00);
	command(start);
	command(scrollInterval);
	command(stop);
	command(0x01); // Vertical scrolling offset
	command(ACTIVATESCROLL);
}

/** \brief Vertical flip.

Flip the graphics on the OLED vertically.
*/
void MicroOLED::flipVertical(boolean flip)
{
	if (flip)
	{
		command(COMSCANINC);
	}
	else
	{
		command(COMSCANDEC);
	}
}

/** \brief Horizontal flip.

    Flip the graphics on the OLED horizontally.
*/
void MicroOLED::flipHorizontal(boolean flip)
{
	if (flip)
	{
		command(SEGREMAP | 0x0);
	}
	else
	{
		command(SEGREMAP | 0x1);
	}
}

/*
	Return a pointer to the start of the RAM screen buffer for direct access.
*/
uint8_t *MicroOLED::getScreenBuffer(void)
{
	return screenmemory;
}

/*
Draw Bitmap image on screen. The array for the bitmap can be stored in the Arduino file, so user don't have to mess with the library files.
To use, create uint8_t array that is 64x48 pixels (384 bytes). Then call .drawBitmap and pass it the array.
*/
void MicroOLED::drawBitmap(uint8_t *bitArray)
{
	for (int i = 0; i < (LCDWIDTH * LCDHEIGHT / 8); i++)
		screenmemory[i] = bitArray[i];
}

//Draw an icon at a given spot
//Use http://en.radzio.dxp.pl/bitmap_converter/ to generate output
//Make sure the bitmap is n*8 pixels tall (pad white pixels to lower area as needed)
//Otherwise the bitmap bitmap_converter will compress some of the bytes together
//TO DO: fix compiler warning re. iconHeight being unused. Maybe use it to check if
//       the icon will fit on the screen?
void MicroOLED::drawIcon(uint8_t offsetX, uint8_t offsetY, uint8_t iconWidth, uint8_t iconHeight, uint8_t *bitArray, uint8_t arraySizeInBytes, bool overwrite)
{
	uint8_t columnNumber = offsetX;
	uint8_t rowNumber = offsetY / 8;

	uint8_t bitOffset = offsetY % 8;

	for (uint8_t i = 0; i < arraySizeInBytes; i++)
	{
		uint16_t byteNumber = rowNumber * 64 + columnNumber;

		//If we have an offset, then straddle the bytes between rows
		if (bitOffset)
		{

			//Zero out the bits before entering new data
			if (overwrite == true)
			{
				screenmemory[byteNumber] &= ~(0xFF << bitOffset);
				if (byteNumber + 64 < (LCDHEIGHT * LCDWIDTH / 8)) //Prevent writes outside screenmemory
					screenmemory[byteNumber + 64] &= ~(0xFF >> (8 - bitOffset));
			}

			//Write in new data across two rows
			screenmemory[byteNumber] |= bitArray[i] << bitOffset;
			if (byteNumber + 64 < (LCDHEIGHT * LCDWIDTH / 8)) //Prevent writes outside screenmemory
				screenmemory[byteNumber + 64] |= bitArray[i] >> (8 - bitOffset);
		}
		else //No shift needed
		{
			if (byteNumber < (LCDHEIGHT * LCDWIDTH) / 8) //Prevent writes outside screenmemory
			{
				//Regular clear
				if (overwrite == true)
					screenmemory[byteNumber] = 0;
				screenmemory[byteNumber] |= bitArray[i];
			}
		}

		columnNumber++;
		if (columnNumber == offsetX + iconWidth)
		{
			//Wrap to next row
			rowNumber++;
			columnNumber = offsetX;
		}
	}
}

//Sets the global size for I2C transactions
//Most platforms use 32 bytes (the default) but this allows users to increase the transaction
//size if the platform supports it
//Note: If the transaction size is set larger than the platforms buffer size, bad things will happen.
void MicroOLED::setI2CTransactionSize(uint8_t transactionSize)
{
  i2cTransactionSize = transactionSize;
}
uint8_t MicroOLED::getI2CTransactionSize(void)
{
  return (i2cTransactionSize);
}

void MicroOLED::swapOLED(uint8_t *x, uint8_t *y)
{
	uint8_t t = *x;
	*x = *y;
	*y = t;
}
