﻿CompilerIf Defined(INCLUDE_MENUCOLORPICKERCLASS, #PB_Constant)=0
#INCLUDE_MENUCOLORPICKERCLASS=1
;/////////////////////////////////////////////////////////////////////////////////
;An OOP class for implementing a menu color picker.

;By Stephen Rodriguez.
;February 2009.

;Developed with Purebasic 4.3.
;/////////////////////////////////////////////////////////////////////////////////

;/////////////////////////////////////////////////////////////////////////////////
;To create an instance of a new menu color picker class, use something like :
;
;   MyMenuColorPickerObject.MenuColorPickerObject
;   MyMenuColorPickerObject = NewMenuColorPicker(basicColorText$, moreColorsText$)   (See demo).
;
;and check the return value is non zero.
;
;/////////////////////////////////////////////////////////////////////////////////

;/////////////////////////////////////////////////////////////////////////////////
;NOTES.
;
; You can place a color picker within any popup menu. The only thing you cannot do is place the color items
; within a submenu.  Other items can go in submenus as per usual.
;/////////////////////////////////////////////////////////////////////////////////

;/////////////////////////////////////////////////////////////////////////////////
;Methods exposed by the MenuColorPicker class :
;
;     Destroy()
;     AddColors(defaultColor=-1)
;     DisplayColorPicker.i(hWnd, x=-1, y=-1)
;     SetUserCallback.i(address)
;/////////////////////////////////////////////////////////////////////////////////


EnableExplicit

XIncludeFile "menuColorPickerClass_Residents.pbi"


;/////////////////////////////////////////////////////////////////////////////////
;-PROTOTYPES.
  Prototype MenuColorPickerClass_UserCallback(row, column, selectedItem, color)
;/////////////////////////////////////////////////////////////////////////////////


;/////////////////////////////////////////////////////////////////////////////////
;-CONSTANTS (internal).

;The following enumeration lists the two menu items which we use. 
;You must ensure that these do not clash with any in your program. 
Enumeration 
  #MenuColorPicker_colors         = 10000
  #MenuColorPicker_morecolors 
EndEnumeration 

#MenuColorPicker_colorwidth = 14
;/////////////////////////////////////////////////////////////////////////////////


;/////////////////////////////////////////////////////////////////////////////////
;-STRUCTURES.

  ;The following structure contains the class template and private properties,
    Structure _membersMenuColorPickerClass
      *vTable.l
      selectedColor.q
      hWnd.i
      oldproc.i 
      userProc.MenuColorPickerClass_UserCallback ; ******* USER CALLBACK ******* 
      menuID.i
      menuhwnd.i 
      selecteditem.i 
      oldselecteditem.i ; ******* USER CALLBACK ******* 
      menuwindowhwnd.i  
      menuBasic$
      menuMore$
      blnColorsAdded.i
      defaultColor.l
      ;Globals.
        hook.i 
        highlightPen.i 
        erasePen.i 
      ;Used for calculations. 
        leftbase.i ;x-coord of left most colour cell. 
        topbase.i ;y-coord of left most colour cell. 
    EndStructure 
;/////////////////////////////////////////////////////////////////////////////////


;/////////////////////////////////////////////////////////////////////////////////
;-GLOBALS.
  Global *mcp_gObjectBeingDisplayed._membersMenuColorPickerClass
;/////////////////////////////////////////////////////////////////////////////////


;/////////////////////////////////////////////////////////////////////////////////
;-INSTANTIATION.
;Returns zero if an error.
;After this call, you must then add the menu items as with regular menus.
Procedure.i NewMenuColorPicker(basicColorsText$ = "Basic colors :", moreColorsText$ = "More colors ...")
  Protected *object._membersMenuColorPickerClass
  Protected menu
  menu=CreatePopupMenu(#PB_Any)
  If menu
    *object = AllocateMemory(SizeOf(_membersMenuColorPickerClass))
    If *object
      With *object
        \vTable = ?VTable_MenuColorPickerClass
        \menuID = menu
        \defaultColor = -1
        \menuBasic$ = basicColorsText$
        \menuMore$ = moreColorsText$
        \highlightPen = CreatePen_(#PS_SOLID, 2, 0) 
        \erasePen = CreatePen_(#PS_SOLID, 2, GetSysColor_(#COLOR_MENU))
        \leftbase = -1
        \topbase = -1
        \menuhwnd=MenuID(menu) 
      EndWith 
    EndIf
  EndIf
  ProcedureReturn *object
EndProcedure
;/////////////////////////////////////////////////////////////////////////////////


;-UTILITY FUNCTIONS.

CompilerIf Defined(FreeStructureString, #PB_Procedure)=0
  ;/////////////////////////////////////////////////////////////////////////////////
  ;The following function frees the string pointed to by *Address.
  ;Used for structures which are created using AllocateMemory().
  Procedure FreeStructureString(*Address) 
    Protected String.String 
    PokeI(@String, *Address)
  EndProcedure
  ;/////////////////////////////////////////////////////////////////////////////////
CompilerEndIf


;/////////////////////////////////////////////////////////////////////////////////
;The following window procedure takes responsibility for drawing the colour cells. 
Procedure MenuColorPicker_CallbackProc(hwnd, uMsg, wParam, lParam) 
  Protected result, *lpmis.MEASUREITEMSTRUCT, *lpdis.DRAWITEMSTRUCT 
  Protected hdc, brush, oldbrush, borderpen, oldpen, i
  Protected Col, row, color , text$
  Select uMsg 
    Case #WM_COMMAND 
      If wParam>>16 & $FFFF=0 
        i = wParam&$FFFF
        Select i
          Case #MenuColorPicker_morecolors
              *mcp_gObjectBeingDisplayed\selectedcolor = ColorRequester(*mcp_gObjectBeingDisplayed\defaultColor) 
          Case #MenuColorPicker_colors
          Default
            *mcp_gObjectBeingDisplayed\selectedcolor = i<<32
          EndSelect
        EndIf 

    Case #WM_MEASUREITEM ;This message is called when the menu / submenu is first created and is used to size the menu items. 
      *lpmis = lParam 
      If *lpmis\CtlType = #ODT_MENU
        Select *lpmis\itemID 
          Case #MenuColorPicker_colors ;The item with the colours. 
            *lpmis\itemWidth = 170 ;Width of menu. 
            *lpmis\itemHeight = 30+12*#MenuColorPicker_colorwidth ;Height of menuitem with the colour cells. 
          Case #MenuColorPicker_morecolors 
            *lpmis\itemWidth = 150 ;Width of menu. 
            *lpmis\itemHeight = 25 ;Height of menuitem with the colour cells. 
          Default
            *lpmis\itemWidth = 150
            *lpmis\itemHeight = 25
        EndSelect 
        result = #True 
      EndIf 

    Case #WM_DRAWITEM  
      *lpdis = lParam 
      If *lpdis\CtlType = #ODT_MENU   ;Don't want this to run for an owner drawn button etc! 
        hdc = *lpdis\hdc 
        Select *lpdis\itemID 
          Case #MenuColorPicker_colors ;The item with the colours. 
            *mcp_gObjectBeingDisplayed\leftbase = (*lpdis\rcItem\right-23*#MenuColorPicker_colorwidth/2)/2 
            *mcp_gObjectBeingDisplayed\topbase = *lpdis\rcItem\top + 30
            *lpdis\rcItem\left+20 : *lpdis\rcItem\top+5 
            DrawText_(hdc, *mcp_gObjectBeingDisplayed\menuBasic$, Len(*mcp_gObjectBeingDisplayed\menuBasic$), *lpdis\rcItem,#DT_SINGLELINE|#DT_LEFT|#DT_TOP|#DT_EXPANDTABS) 
            *lpdis\rcItem\left-20 : *lpdis\rcItem\top-5 
            For row = 0 To 7 
              For Col = 0 To 7 
                i = row*8 + Col
;                brush=CreateSolidBrush_(RGB((i>>4)%2*$FF,i%16*$11,i>>5*$FF)) 
                brush=CreateSolidBrush_(RGB(i%4*$55,(i%16)>>2*$55,i>>4*$55)) 
                oldbrush=SelectObject_(hdc,brush) 
                Rectangle_(hdc, *mcp_gObjectBeingDisplayed\leftbase+Col*3*#MenuColorPicker_colorwidth/2, *mcp_gObjectBeingDisplayed\topbase+row*3*#MenuColorPicker_colorwidth/2, *mcp_gObjectBeingDisplayed\leftbase+Col*3*#MenuColorPicker_colorwidth/2+#MenuColorPicker_colorwidth-1,*mcp_gObjectBeingDisplayed\topbase+row*3*#MenuColorPicker_colorwidth/2+#MenuColorPicker_colorwidth-1) 
                SelectObject_(hdc,oldbrush) 
                DeleteObject_(brush) 
              Next 
            Next 
        EndSelect 
        result = #True 
      EndIf 
    Default 
      If *mcp_gObjectBeingDisplayed\oldproc 
        ProcedureReturn CallWindowProc_(*mcp_gObjectBeingDisplayed\oldproc, hwnd, uMsg, wParam, lParam) 
      EndIf 
  EndSelect 
  ProcedureReturn result 
EndProcedure 
;/////////////////////////////////////////////////////////////////////////////////


;/////////////////////////////////////////////////////////////////////////////////
;The following hook procedure follows the mouse whilst the MenuColour is visible, displaying selection 
;borders as appropriate. 
Procedure.i MenuColorPicker_MouseProc(nCode, wParam, lParam)  
  Protected *mp.MOUSEHOOKSTRUCT, rc.RECT,leftdis, topdis, selecteditem, row, Col 
  Protected hdc, i, j, oldpen, oldbrush, flag, color, wndhWnd, pt.POINT
  If nCode >=0 And wParam = #WM_MOUSEMOVE 
    *mp =lParam 
    wndhWnd = WindowFromPoint_(*mp\pt\y<<32 + *mp\pt\x)
    If *mcp_gObjectBeingDisplayed\menuwindowhwnd=0 
      GetMenuItemRect_(0, *mcp_gObjectBeingDisplayed\menuhWnd, 0, rc)
      pt\x = rc\left
      pt\y = rc\top
      *mcp_gObjectBeingDisplayed\menuwindowhwnd=WindowFromPoint_(pt\y<<32 + pt\x)
    EndIf 
    If IsWindow_(*mcp_gObjectBeingDisplayed\menuwindowhwnd) And (wndhWnd = *mcp_gObjectBeingDisplayed\menuwindowhwnd Or *mcp_gObjectBeingDisplayed\selecteditem>-1)
      CopyMemory(*mp\pt, pt, SizeOf(POINT))
      MapWindowPoints_(0, *mcp_gObjectBeingDisplayed\menuwindowhwnd, pt, 1)
      ;Calculations required for determining the coordinates of individual colour cells. 
        leftdis=(pt\x-*mcp_gObjectBeingDisplayed\leftbase) 
        topdis=(pt\y-*mcp_gObjectBeingDisplayed\topbase) 
        row=topdis/(3*#MenuColorPicker_colorwidth/2) : Col = leftdis/(3*#MenuColorPicker_colorwidth/2) 
      ;Determine if the cursor is over a colour item. 
        If leftdis >= 0 And topdis >= 0 And leftdis%(3*#MenuColorPicker_colorwidth/2)<#MenuColorPicker_colorwidth And leftdis%(3*#MenuColorPicker_colorwidth/2)>=0 And topdis%(3*#MenuColorPicker_colorwidth/2)<#MenuColorPicker_colorwidth And topdis%(3*#MenuColorPicker_colorwidth/2)>=0 And row < 8 And Col < 8
          selecteditem = row*8 + Col  
        Else
          selecteditem = -1
        EndIf 
      If selecteditem <> *mcp_gObjectBeingDisplayed\selecteditem
        hdc=GetDC_(*mcp_gObjectBeingDisplayed\menuwindowhwnd) 
        If hdc 
          oldbrush=SelectObject_(hdc,GetStockObject_(#NULL_BRUSH)) 
          oldpen=SelectObject_(hdc, *mcp_gObjectBeingDisplayed\erasepen) 
          If *mcp_gObjectBeingDisplayed\selecteditem > -1
            i = *mcp_gObjectBeingDisplayed\selecteditem>>3
            j = *mcp_gObjectBeingDisplayed\selecteditem%8
            Rectangle_(hdc, *mcp_gObjectBeingDisplayed\leftbase+j*3*#MenuColorPicker_colorwidth/2-2, *mcp_gObjectBeingDisplayed\topbase+i*3*#MenuColorPicker_colorwidth/2-2, *mcp_gObjectBeingDisplayed\leftbase+j*3*#MenuColorPicker_colorwidth/2+#MenuColorPicker_colorwidth+2,*mcp_gObjectBeingDisplayed\topbase+i*3*#MenuColorPicker_colorwidth/2+#MenuColorPicker_colorwidth+2) 
          EndIf
          If selecteditem > -1
            SelectObject_(hdc, *mcp_gObjectBeingDisplayed\highlightpen) 
            Rectangle_(hdc, *mcp_gObjectBeingDisplayed\leftbase+col*3*#MenuColorPicker_colorwidth/2-2, *mcp_gObjectBeingDisplayed\topbase+row*3*#MenuColorPicker_colorwidth/2-2, *mcp_gObjectBeingDisplayed\leftbase+col*3*#MenuColorPicker_colorwidth/2+#MenuColorPicker_colorwidth+2,*mcp_gObjectBeingDisplayed\topbase+row*3*#MenuColorPicker_colorwidth/2+#MenuColorPicker_colorwidth+2) 
            SelectObject_(hdc,*mcp_gObjectBeingDisplayed\erasepen) 
            *mcp_gObjectBeingDisplayed\selectedcolor = RGB(selecteditem%4*$55,(selecteditem%16)>>2*$55,selecteditem>>4*$55)
          Else
            *mcp_gObjectBeingDisplayed\selectedcolor = -1
          EndIf
          ; ******* USER CALLBACK ******* 
            If *mcp_gObjectBeingDisplayed\userProc 
              *mcp_gObjectBeingDisplayed\userProc(row+1, Col+1, selecteditem+1, *mcp_gObjectBeingDisplayed\selectedcolor) 
            EndIf 
          ; ******* USER CALLBACK ******* 
        EndIf
        *mcp_gObjectBeingDisplayed\selecteditem = selecteditem
      EndIf
    EndIf
  EndIf 
  ProcedureReturn CallNextHookEx_(*mcp_gObjectBeingDisplayed\hook, nCode, wParam, lParam) 
EndProcedure 
;/////////////////////////////////////////////////////////////////////////////////


;-CLASS METHODS.

;/////////////////////////////////////////////////////////////////////////////////
;The following method frees all memory associated with the underlying object.
Procedure menuColorPickerClass_Destroy(*this._membersMenuColorPickerClass)
  ;Free structured strings.
    FreeStructureString(@*this\menuBasic$)
    FreeStructureString(@*this\menuMore$)
  ;Delete pens.
    DeleteObject_(*this\highlightPen)
    DeleteObject_(*this\erasePen)
  ;Free menu.
    FreeMenu(*this\menuID)
  FreeMemory(*this)
EndProcedure
;/////////////////////////////////////////////////////////////////////////////////


;/////////////////////////////////////////////////////////////////////////////////
Procedure menuColorPickerClass_AddColors(*this._membersMenuColorPickerClass)
  If *this\blnColorsAdded = #False
    InsertMenu_(*this\menuhWnd, -1, #MF_BYPOSITION|#MF_STRING, #MenuColorPicker_colors, @"")
    ModifyMenu_(*this\menuhwnd,#MenuColorPicker_colors,#MF_BYCOMMAND|#MF_OWNERDRAW,#MenuColorPicker_colors,0) 
    InsertMenu_(*this\menuhWnd, -1, #MF_BYPOSITION|#MF_SEPARATOR, 0, 0)
    InsertMenu_(*this\menuhWnd, -1, #MF_BYPOSITION|#MF_STRING, #MenuColorPicker_morecolors, @*this\menuMore$)
;    ModifyMenu_(*this\menuhwnd,#MenuColorPicker_morecolors,#MF_BYCOMMAND|#MF_OWNERDRAW,#MenuColorPicker_morecolors,0) 
    *this\blnColorsAdded = #True
  EndIf
EndProcedure
;/////////////////////////////////////////////////////////////////////////////////


;/////////////////////////////////////////////////////////////////////////////////
;Returns a menuitemid or -1 if the menu was cancelled (or an error) or the selected color.
Procedure.q menuColorPickerClass_DisplayColorPicker(*this._membersMenuColorPickerClass, hWnd, x=-1, y=-1)
  Protected menu, hInstance, lpdwProcessId
  If hWnd
    *this\selecteditem =-1 
    *this\oldselecteditem = -1 
    *this\selectedColor = -1
    *this\hWnd = hWnd
    *mcp_gObjectBeingDisplayed = *this
    *this\menuwindowhwnd = 0
    ;Need to subclass the window.
      *this\oldproc=SetWindowLongPtr_(hWnd, #GWL_WNDPROC, @MenuColorPicker_CallbackProc()) 
    hInstance = GetModuleHandle_(0) 
    lpdwProcessId = GetWindowThreadProcessId_(hWnd, 0) 
    *this\hook= SetWindowsHookEx_(#WH_MOUSE, @MenuColorPicker_MouseProc(), hInstance, lpdwProcessId) 
    If x=-1 Or y = -1 
      DisplayPopupMenu(*this\menuID, hWnd) 
    Else 
      DisplayPopupMenu(*this\menuID, hWnd,x,y) 
    EndIf 
    UnhookWindowsHookEx_(*this\hook) 
    WindowEvent() ;Just in case the 'More colors...' option was clicked. 
    ;Reinstate the original window proc.
      SetWindowLongPtr_(hWnd, #GWL_WNDPROC, *this\oldproc)
    *this\oldproc = 0 
    ProcedureReturn *this\selectedcolor
  EndIf
EndProcedure
;/////////////////////////////////////////////////////////////////////////////////


;/////////////////////////////////////////////////////////////////////////////////
;The following method frees all memory associated with the underlying object.
Procedure.i menuColorPickerClass_SetDefaultColor(*this._membersMenuColorPickerClass, color)
  *this\defaultColor = color
EndProcedure
;/////////////////////////////////////////////////////////////////////////////////


;/////////////////////////////////////////////////////////////////////////////////
;The following method frees all memory associated with the underlying object.
Procedure.i menuColorPickerClass_SetUserCallback(*this._membersMenuColorPickerClass, address)
  Protected result
  result = *this\userProc
  If address
    *this\userProc = address
  EndIf
  ProcedureReturn result
EndProcedure
;/////////////////////////////////////////////////////////////////////////////////


;-VIRTUAL TABLES.

DataSection 
  VTable_MenuColorPickerClass: 
    Data.i @menuColorPickerClass_Destroy()
    Data.i @menuColorPickerClass_AddColors()
    Data.i @menuColorPickerClass_DisplayColorPicker()
    Data.i @menuColorPickerClass_SetDefaultColor()
    Data.i @menuColorPickerClass_SetUserCallback()
EndDataSection 

DisableExplicit

CompilerEndIf

; IDE Options = PureBasic 4.30 (Windows - x86)
; CursorPosition = 331
; FirstLine = 328
; Folding = --