; Author: Kelebrindae
; Date: December, 15, 2010
; PB version: v4.51
; ---------------------------------------------------------------------------------------------------------------
; Description:
; ---------------------------------------------------------------------------------------------------------------
; A simple 2D physics engine, based on Verlet integration. For a physics noob like me, Verlet looks like magic!
;
; The physics code is based on the following two articles:
;   - http://www.gamasutra.com/resource_guide/20030121/jacobson_01.shtml
;   - http://www.gamedev.net/reference/articles/article2714.asp
; ---------------------------------------------------------------------------------------------------------------
; Known bugs and limitations:
; ---------------------------------------------------------------------------------------------------------------
; - Considering my knowledge of physics and my fluency with vectors, most of the maths in the collision response 
;   are probably wrong. Sorry!
; - TO DO: Procs to weld / separate two points
; - TO DO: Ball simulation (by adding a radius to a single point)
; - TO DO: Adding a "stop" state to bodies under special conditions (near-to-zero speed, resting on an object...)
;          to increase stability.
; ---------------------------------------------------------------------------------------------------------------

;- Constants, structures, globals
; This constant is used to read the lists of points/constraints in memory 
CompilerIf #PB_Compiler_Processor = #PB_Processor_x64
  #SIZEOFPTR = 8
CompilerElse
  #SIZEOFPTR = 4
CompilerEndIf


#SUBSAMPLING = 5              ; Greater sub-sampling means slower but more precise simulation (and objects that don't pass through each others)
#CONSTRAINT_ITERATIONS = 4    ; More iterations means that bodies are more rigid
#GRAVITY = 0.005              ; Should be set according to #SUBSAMPLING (gravity is applied in each sub-sampling iteration)
#FRICTION = 0.2               ; Friction for a point colliding against an edge
#FRICTION_EDGEONPOINT = 0.01  ; Friction for a edge colliding against a single point

; 2D vector macro library
IncludeFile "vector2.pbi"

; Pointmass, a.k.a. vertices of a rigid body
Structure pointmass_struct
  x.f     ; Coords x,y
  y.f 
  oldX.f  ; Previous coords x,y
  oldY.f  ; in Verlet integration, velocity, momentum, etc.. are implicit: we just need current and previous coords.
  mass.f    ; mass of the point
  invmass.f ; inverse mass. Set it to zero for static points
EndStructure
Global NewList pointmass.pointmass_struct()

; Constraints, a.k.a. edges of a rigid body
Structure constraint_struct
 *p1.pointmass_struct ; pointer to first point
 *p2.pointmass_struct ; pointer to second point
 enable.b             ; will this constraint be checked for collision ?
 length.f             ; length of the constraint
 length2.f            ; this is equals to length*length (optimization)
 *ptrParent.rigidBody_struct  ; pointer to the parent body
EndStructure
Global NewList constraint.constraint_struct()

; Rigid body
Structure rigidBody_struct
  numint.i  ; Unique ID
  
  nbPoint.i ; Nb points in this body
  nbEdge.i  ; Nb constraints in this body
  *ptrPointList.pointmass_struct        ; pointer to a list of pointers to pointmasses (complicated, but faster than a list() )
  *ptrConstraintList.constraint_struct  ; pointer to a list of pointers to constraints (complicated, but faster than a list() )
  
  ; Point to the parent compound
  *ptrParent.compound_struct
  
  ; These are re-calculated at each frame (for collisions)
  center.vector2_struct
  minX.f
  maxX.f
  minY.f
  maxY.f
EndStructure
Global NewList body.rigidBody_struct()

; Compounds are multi-parts objects, like ropes or bridges. Objects which are part of the same compound don't collide with each other
Structure compound_struct
  name.s
EndStructure
Global NewList compound.compound_struct()

; Used to store collisions datas
Structure collisionInfo_struct
  depth.f
  normal.vector2_struct
  collisionVector.vector2_struct
  
  *ptrEdge.constraint_struct
  *ptrPoint.pointmass_struct
  
  collisionCase.b ; store the collision configuration: no static point, one static point, etc.
EndStructure
Global VERLET_collisionInfo.collisionInfo_struct

; Global vars used in Macros
Global VERLET_i.i,VERLET_j.i,VERLET_numBody.i,VERLET_nbEdges.i
Global VERLET_dx.f,VERLET_dy.f, VERLET_deltalength.f,VERLET_diff.f
Global *VERLET_ptrPoint.pointmass_struct,*VERLET_ptrEdge.constraint_struct,*VERLET_ptrGen,*VERLET_ptrGen2
Global VERLET_axis.vector2_struct,VERLET_tempVector.vector2_struct
Global VERLET_minA.f,VERLET_minB.f,VERLET_maxA.f,VERLET_maxB.f,VERLET_distance.f,VERLET_minDistance.f
Global VERLET_dotP.f
Global VERLET_ratio1.f,VERLET_ratio3.f
Global *VERLET_ptrCollPointBody.rigidBody_struct,*VERLET_ptrCollEdgeBody.rigidBody_struct
Global VERLET_T.f,VERLET_lambda.f,VERLET_edgeMass.f,VERLET_invCollisionMass.f
Global *VERLET_prevBodyPtr.rigidBody_struct, VERLET_startIndex.i, VERLET_isCollision.b


;- Collision and movements

;************************************************************************************
; Name: INTERVAl_DISTANCE
; Purpose: Indicates the minimum distance between the points of two overlapping segments
; Parameters:
;   - segment A min (float)
;   - segment A max (float)
;   - segment B min (float)
;   - segment B max (float)
;   - distance (output float)
;************************************************************************************
Macro INTERVAL_DISTANCE(minA,maxA,minB,maxB,distance)
  If minA < minB
    distance = maxA - minB
  Else
    distance = maxB - minA
  EndIf
EndMacro

;************************************************************************************
; Name: PROJECT_TO_AXIS
; Purpose: Projects the points of a body onto an line
; Parameters:
;   - pointer to a rigid body (*ptr)
;   - axis (vector2)
;   - min (output float)
;   - max (output float)
;************************************************************************************
Macro PROJECT_TO_AXIS (ptrBody,axis,minValue,maxValue)
  *VERLET_ptrGen = ptrBody\ptrPointList
  *VERLET_ptrPoint = PeekI(*VERLET_ptrGen)
  VERLET_dotP = VEC2_dotProduct(axis,*VERLET_ptrPoint)
  minValue = VERLET_dotP
  maxValue = VERLET_dotP
  For VERLET_j = 1 To ptrBody\nbPoint
    *VERLET_ptrGen + #SIZEOFPTR
    *VERLET_ptrPoint = PeekI(*VERLET_ptrGen)
    
    ; Project the rest of the vertices onto the axis and extend the interval to the left/right if necessary
    VERLET_dotP = VEC2_dotProduct(axis,*VERLET_ptrPoint)
    If VERLET_dotP < minValue
      minValue = VERLET_dotP
    EndIf
    If VERLET_dotP > maxValue
      maxValue = VERLET_dotP
    EndIf
  Next VERLET_j
EndMacro    

;************************************************************************************
; Name: COLLISION_RESPONSE
; Purpose: Computes the collision response between a point and a segment. None, one or
;          two points may be static (which leads to different response managements)
; Parameters:
;   None => the needed info is stored in the "VERLET_collisionInfo" structure
;************************************************************************************
Macro COLLISION_RESPONSE()
    ; Collision response = VERLET_collisionInfo\normal * VERLET_collisionInfo\depth
  VEC2_multiply(VERLET_collisionInfo\normal,VERLET_collisionInfo\depth,VERLET_collisionInfo\collisionVector)
  
  ; Edge collision response: we need to know where the point hit the edge
  If( Abs( VERLET_collisionInfo\ptrEdge\p1\x - VERLET_collisionInfo\ptrEdge\p2\x ) > Abs( VERLET_collisionInfo\ptrEdge\p1\y - VERLET_collisionInfo\ptrEdge\p2\y ) )
    VERLET_T = ( VERLET_collisionInfo\ptrPoint\x - VERLET_collisionInfo\collisionVector\x - VERLET_collisionInfo\ptrEdge\p1\x )/(  VERLET_collisionInfo\ptrEdge\p2\x - VERLET_collisionInfo\ptrEdge\p1\x)
  Else
    VERLET_T = ( VERLET_collisionInfo\ptrPoint\y - VERLET_collisionInfo\collisionVector\y - VERLET_collisionInfo\ptrEdge\p1\y )/(  VERLET_collisionInfo\ptrEdge\p2\y - VERLET_collisionInfo\ptrEdge\p1\y)
  EndIf
  VERLET_lambda = 1.0 / ( VERLET_T*VERLET_T + ( 1 - VERLET_T )*( 1 - VERLET_T ) )
  
; Friction of collision with a dynamic body is a little more tricky. You need to calculate relative velocity of colliding points. 
; Let's assume your mass point I collided with and edge (defined by mass points J and K) of the other object. 
;  1 - The relative velocity is relVel = I.pos - I.last_pos - (J.pos + K.pos - J.last_pos - K.last_pos)/2. 
;  2 - you compute tangent from the collision normal and project the relative velocity onto it: relTanVel = tangent*dot(relVel, tangent)
;  3 - you add/subtract (depending on how you computed tangent) some fraction of relTanVel to/from last_pos of I,J,K. 
;      What fraction it will be is determined by I,J,K masses and "hit coefficient" (because I is somewhere between J and K...you get the hit coefficient by projecting I onto the edge JK).

  ; From here, we need the absolute value of the collision normal
  If VERLET_collisionInfo\normal\x < 0
    VERLET_collisionInfo\normal\x = -VERLET_collisionInfo\normal\x
  EndIf
  If VERLET_collisionInfo\normal\y < 0
    VERLET_collisionInfo\normal\y = -VERLET_collisionInfo\normal\y
  EndIf  
  
  ; What's configuration are we in (is there some staic points involved?)
  If VERLET_collisionInfo\ptrPoint\invMass = 0
    VERLET_collisionInfo\collisionCase = %001
  Else
    VERLET_collisionInfo\collisionCase = %000
  EndIf
  If VERLET_collisionInfo\ptrEdge\p1\invmass = 0
    VERLET_collisionInfo\collisionCase + %010
  EndIf
  If VERLET_collisionInfo\ptrEdge\p2\invmass = 0
    VERLET_collisionInfo\collisionCase + %100
  EndIf
  
  ; CASE 1 : no static point
  Select VERLET_collisionInfo\collisionCase
    Case %000 ; no Static point
      ; Mass of the constraint at the intersection point
      VERLET_edgeMass = VERLET_T * VERLET_collisionInfo\ptrEdge\p2\Mass + ( 1 - VERLET_T ) * VERLET_collisionInfo\ptrEdge\p1\Mass
      VERLET_invCollisionMass = 1/( VERLET_edgeMass + VERLET_collisionInfo\ptrPoint\Mass )
      
      ; Response repartition, according to each point's mass
      VERLET_ratio1 = VERLET_collisionInfo\ptrPoint\Mass * VERLET_invCollisionMass
      VERLET_ratio3 = VERLET_edgeMass * VERLET_invCollisionMass;
      
      ; Constraint collision response
      VERLET_collisionInfo\ptrEdge\p1\x + VERLET_collisionInfo\collisionVector\x * ( 1 - VERLET_T ) * VERLET_ratio1 * VERLET_lambda
      VERLET_collisionInfo\ptrEdge\p1\y + VERLET_collisionInfo\collisionVector\y * ( 1 - VERLET_T ) * VERLET_ratio1 * VERLET_lambda
      
      VERLET_collisionInfo\ptrEdge\p2\x + VERLET_collisionInfo\collisionVector\x * VERLET_T  * VERLET_ratio1 * VERLET_lambda
      VERLET_collisionInfo\ptrEdge\p2\y + VERLET_collisionInfo\collisionVector\y * VERLET_T  * VERLET_ratio1 * VERLET_lambda
      
      ; Constraint friction
      VERLET_collisionInfo\ptrEdge\p1\oldX + (VERLET_collisionInfo\ptrEdge\p1\x - VERLET_collisionInfo\ptrEdge\p1\oldX) * VERLET_collisionInfo\normal\y * #FRICTION_EDGEONPOINT
      VERLET_collisionInfo\ptrEdge\p1\oldY + (VERLET_collisionInfo\ptrEdge\p1\y - VERLET_collisionInfo\ptrEdge\p1\oldY) * VERLET_collisionInfo\normal\x * #FRICTION_EDGEONPOINT
      VERLET_collisionInfo\ptrEdge\p2\oldX + (VERLET_collisionInfo\ptrEdge\p2\x - VERLET_collisionInfo\ptrEdge\p2\oldX) * VERLET_collisionInfo\normal\y * #FRICTION_EDGEONPOINT
      VERLET_collisionInfo\ptrEdge\p2\oldY + (VERLET_collisionInfo\ptrEdge\p2\y - VERLET_collisionInfo\ptrEdge\p2\oldY) * VERLET_collisionInfo\normal\x * #FRICTION_EDGEONPOINT
      
      ; Vertex collision response
      VERLET_collisionInfo\ptrPoint\x - VERLET_collisionInfo\collisionVector\x * VERLET_ratio3
      VERLET_collisionInfo\ptrPoint\y - VERLET_collisionInfo\collisionVector\y * VERLET_ratio3
      
      ; Vertex friction
      VERLET_collisionInfo\ptrPoint\oldX + (VERLET_collisionInfo\ptrPoint\x - VERLET_collisionInfo\ptrPoint\oldX) * VERLET_collisionInfo\normal\y * #FRICTION
      
    Case %110 ; point vs. static constraint
      ; Vertex collision response
      VERLET_collisionInfo\ptrPoint\x - VERLET_collisionInfo\collisionVector\x
      VERLET_collisionInfo\ptrPoint\y - VERLET_collisionInfo\collisionVector\y
      
      ; Vertex friction
      VERLET_collisionInfo\ptrPoint\oldX + (VERLET_collisionInfo\ptrPoint\x - VERLET_collisionInfo\ptrPoint\oldX) * VERLET_collisionInfo\normal\y * #FRICTION
      VERLET_collisionInfo\ptrPoint\oldY + (VERLET_collisionInfo\ptrPoint\y - VERLET_collisionInfo\ptrPoint\oldY) * VERLET_collisionInfo\normal\x * #FRICTION
      
    Case %001 ; constraint vs. static point
      VERLET_ratio1 = 1 / (VERLET_T * VERLET_collisionInfo\ptrEdge\p2\Mass + ( 1 - VERLET_T ) * VERLET_collisionInfo\ptrEdge\p1\Mass)
      
      ; Constraint collision response
      VERLET_collisionInfo\ptrEdge\p1\x + VERLET_collisionInfo\collisionVector\x * ( 1 - VERLET_T ) * VERLET_lambda * VERLET_ratio1
      VERLET_collisionInfo\ptrEdge\p1\y + VERLET_collisionInfo\collisionVector\y * ( 1 - VERLET_T ) * VERLET_lambda * VERLET_ratio1
      
      VERLET_collisionInfo\ptrEdge\p2\x + VERLET_collisionInfo\collisionVector\x * VERLET_T * VERLET_lambda * VERLET_ratio1
      VERLET_collisionInfo\ptrEdge\p2\y + VERLET_collisionInfo\collisionVector\y * VERLET_T * VERLET_lambda * VERLET_ratio1
      
      ; Constraint friction
      VERLET_collisionInfo\ptrEdge\p1\oldX + (VERLET_collisionInfo\ptrEdge\p1\x - VERLET_collisionInfo\ptrEdge\p1\oldX) * VERLET_collisionInfo\normal\y * #FRICTION_EDGEONPOINT
      VERLET_collisionInfo\ptrEdge\p1\oldY + (VERLET_collisionInfo\ptrEdge\p1\y - VERLET_collisionInfo\ptrEdge\p1\oldY) * VERLET_collisionInfo\normal\x * #FRICTION_EDGEONPOINT
      VERLET_collisionInfo\ptrEdge\p2\oldX + (VERLET_collisionInfo\ptrEdge\p2\x - VERLET_collisionInfo\ptrEdge\p2\oldX) * VERLET_collisionInfo\normal\y * #FRICTION_EDGEONPOINT
      VERLET_collisionInfo\ptrEdge\p2\oldY + (VERLET_collisionInfo\ptrEdge\p2\y - VERLET_collisionInfo\ptrEdge\p2\oldY) * VERLET_collisionInfo\normal\x * #FRICTION_EDGEONPOINT
      
    Case %010 ; point vs. semi-static constraint (first point is static)
      VERLET_edgeMass = ( 1 - VERLET_T ) * VERLET_collisionInfo\ptrEdge\p2\Mass 
      VERLET_invCollisionMass = 1/( VERLET_edgeMass + VERLET_collisionInfo\ptrPoint\Mass )
      
      VERLET_ratio1 = VERLET_collisionInfo\ptrPoint\Mass * VERLET_invCollisionMass
      VERLET_collisionInfo\ptrEdge\p2\x + VERLET_collisionInfo\collisionVector\x * ( 1 - VERLET_T ) * VERLET_lambda * VERLET_ratio1
      VERLET_collisionInfo\ptrEdge\p2\y + VERLET_collisionInfo\collisionVector\y * ( 1 - VERLET_T ) * VERLET_lambda * VERLET_ratio1
      
      ; Constraint friction
      VERLET_collisionInfo\ptrEdge\p2\oldX + (VERLET_collisionInfo\ptrEdge\p2\x - VERLET_collisionInfo\ptrEdge\p2\oldX) * VERLET_collisionInfo\normal\y * #FRICTION_EDGEONPOINT
      VERLET_collisionInfo\ptrEdge\p2\oldY + (VERLET_collisionInfo\ptrEdge\p2\y - VERLET_collisionInfo\ptrEdge\p2\oldY) * VERLET_collisionInfo\normal\x * #FRICTION_EDGEONPOINT
    
      ; Vertex collision response
      VERLET_ratio3 = VERLET_edgeMass * VERLET_invCollisionMass
      VERLET_collisionInfo\ptrPoint\x - VERLET_collisionInfo\collisionVector\x * VERLET_ratio3
      VERLET_collisionInfo\ptrPoint\y - VERLET_collisionInfo\collisionVector\y * VERLET_ratio3
      
      ; Vertex friction
      VERLET_collisionInfo\ptrPoint\oldX + (VERLET_collisionInfo\ptrPoint\x - VERLET_collisionInfo\ptrPoint\oldX) * VERLET_collisionInfo\normal\y * #FRICTION
      VERLET_collisionInfo\ptrPoint\oldY + (VERLET_collisionInfo\ptrPoint\y - VERLET_collisionInfo\ptrPoint\oldY) * VERLET_collisionInfo\normal\x * #FRICTION
      
    Case %100 ; point vs. semi-static constraint (second point is static)
      ; Constraint collision response
      VERLET_edgeMass = VERLET_T * VERLET_collisionInfo\ptrEdge\p1\Mass
      VERLET_invCollisionMass = 1/( VERLET_edgeMass + VERLET_collisionInfo\ptrPoint\Mass )
      
      VERLET_ratio1 = VERLET_collisionInfo\ptrPoint\Mass * VERLET_invCollisionMass
      VERLET_collisionInfo\ptrEdge\p1\x + VERLET_collisionInfo\collisionVector\x * VERLET_T * VERLET_lambda * VERLET_ratio1
      VERLET_collisionInfo\ptrEdge\p1\y + VERLET_collisionInfo\collisionVector\y * VERLET_T * VERLET_lambda * VERLET_ratio1
      
      ; Constraint friction
      VERLET_collisionInfo\ptrEdge\p1\oldX + (VERLET_collisionInfo\ptrEdge\p1\x - VERLET_collisionInfo\ptrEdge\p1\oldX) * VERLET_collisionInfo\normal\y * #FRICTION_EDGEONPOINT
      VERLET_collisionInfo\ptrEdge\p1\oldY + (VERLET_collisionInfo\ptrEdge\p1\y - VERLET_collisionInfo\ptrEdge\p1\oldY) * VERLET_collisionInfo\normal\x * #FRICTION_EDGEONPOINT
    
      ; Vertex collision response
      VERLET_ratio3 = VERLET_edgeMass * VERLET_invCollisionMass
      VERLET_collisionInfo\ptrPoint\x - VERLET_collisionInfo\collisionVector\x * VERLET_ratio3
      VERLET_collisionInfo\ptrPoint\y - VERLET_collisionInfo\collisionVector\y * VERLET_ratio3
      
      ; Vertex friction
      VERLET_collisionInfo\ptrPoint\oldX + (VERLET_collisionInfo\ptrPoint\x - VERLET_collisionInfo\ptrPoint\oldX) * VERLET_collisionInfo\normal\y * #FRICTION
      VERLET_collisionInfo\ptrPoint\oldY + (VERLET_collisionInfo\ptrPoint\y - VERLET_collisionInfo\ptrPoint\oldY) * VERLET_collisionInfo\normal\x * #FRICTION
      
    Case %011 ; static point vs. semi-static constraint (first point is Static)
      ; Constraint collision response
      VERLET_ratio1 = 1 / (VERLET_T * VERLET_collisionInfo\ptrEdge\p2\Mass )
      VERLET_collisionInfo\ptrEdge\p2\x + VERLET_collisionInfo\collisionVector\x * VERLET_T * VERLET_lambda * VERLET_ratio1
      VERLET_collisionInfo\ptrEdge\p2\y + VERLET_collisionInfo\collisionVector\y * VERLET_T * VERLET_lambda * VERLET_ratio1
      
      ; Constraint friction
      VERLET_collisionInfo\ptrEdge\p2\oldX + (VERLET_collisionInfo\ptrEdge\p2\x - VERLET_collisionInfo\ptrEdge\p2\oldX) * VERLET_collisionInfo\normal\y * #FRICTION_EDGEONPOINT
      VERLET_collisionInfo\ptrEdge\p2\oldY + (VERLET_collisionInfo\ptrEdge\p2\y - VERLET_collisionInfo\ptrEdge\p2\oldY) * VERLET_collisionInfo\normal\x * #FRICTION_EDGEONPOINT
      
    Case %101 ; static point vs. semi-static constraint (second point is Static)
      ; Constraint collision response
      VERLET_ratio1 = 1 / (( 1 - VERLET_T ) * VERLET_collisionInfo\ptrEdge\p1\Mass)
      VERLET_collisionInfo\ptrEdge\p1\x + VERLET_collisionInfo\collisionVector\x * ( 1 - VERLET_T ) * VERLET_lambda * VERLET_ratio1
      VERLET_collisionInfo\ptrEdge\p1\y + VERLET_collisionInfo\collisionVector\y * ( 1 - VERLET_T ) * VERLET_lambda * VERLET_ratio1
      
      ; Constraint friction
      VERLET_collisionInfo\ptrEdge\p1\oldX + (VERLET_collisionInfo\ptrEdge\p1\x - VERLET_collisionInfo\ptrEdge\p1\oldX) * VERLET_collisionInfo\normal\y * #FRICTION_EDGEONPOINT
      VERLET_collisionInfo\ptrEdge\p1\oldY + (VERLET_collisionInfo\ptrEdge\p1\y - VERLET_collisionInfo\ptrEdge\p1\oldY) * VERLET_collisionInfo\normal\x * #FRICTION_EDGEONPOINT
      
  EndSelect
  
EndMacro

;************************************************************************************
; Name: DETECT_COLLISION
; Purpose: Determines if two rigid bodies are colliding, using SAT algorithm
;           http://www.gamedev.net/reference/articles/article2714.asp
; Parameters:
;   - pointer to first body
;   - pointer to second body
;   - collision or not (output boolean)
; Note: This macro looks for a collision between a point and an edge (= a constraint).
;       The structure "collisionInfo" stores the pointer to these, as well as the
;       collsion depth, the collision normal, etc..
;************************************************************************************
Macro DETECT_COLLISION(ptrBody1,ptrBody2,collisionResult)
  
  ; We check all edges of the two bodies
  VERLET_nbEdges = ptrBody1\nbEdge + ptrBody2\nbEdge + 1
  *VERLET_ptrGen2 = ptrBody1\ptrConstraintList
  VERLET_minDistance = 10000000
  collisionResult = #True
  For VERLET_i = 0 To VERLET_nbEdges
    *VERLET_ptrEdge = PeekI(*VERLET_ptrGen2)
    ; When all of body1's edges have been checked, switch to body2
    If VERLET_i = ptrBody1\nbEdge
      *VERLET_ptrGen2 = ptrBody2\ptrConstraintList
    Else
      *VERLET_ptrGen2 + #SIZEOFPTR
    EndIf
    
    If *VERLET_ptrEdge\enable = #False
      Continue
    EndIf
    
    ; Calculate the axis perpendicular (cross vector) to this edge and normalize it
    VERLET_axis\x = *VERLET_ptrEdge\p1\y - *VERLET_ptrEdge\p2\y
    VERLET_axis\y = *VERLET_ptrEdge\p2\x - *VERLET_ptrEdge\p1\x
    VEC2_Normalize(VERLET_axis,VERLET_axis)
    
    ; Project each point of the 2 bodies on this vector; Get min/max for each body
    PROJECT_TO_AXIS(ptrBody1,VERLET_axis,VERLET_minA,VERLET_maxA)
    PROJECT_TO_AXIS(ptrBody2,VERLET_axis,VERLET_minB,VERLET_maxB)
    
    ; Calculate the distance between the two intervals
    INTERVAL_DISTANCE( VERLET_minA, VERLET_maxA, VERLET_minB, VERLET_maxB, VERLET_distance)
       
    ; If the intervals don't overlap, there is no collision
    If VERLET_distance < 0.0       
      collisionResult = #False
      Break
    EndIf     
    
    ; Note: if body contains parallel lines, the "minDistance" edge might not be the good one...
    ; => if distance = minDistance, we should check if the edge is closer to the second body
    If VERLET_distance < VERLET_minDistance
      VERLET_minDistance = VERLET_distance
      VERLET_collisionInfo\normal\x = VERLET_axis\x
      VERLET_collisionInfo\normal\y = VERLET_axis\y
      VERLET_collisionInfo\ptrEdge = *VERLET_ptrEdge
    EndIf
  
  Next VERLET_i
  
  ; If the two bodies collide, what do we do ?
  If collisionResult = #True
    VERLET_collisionInfo\depth = VERLET_minDistance
    
    ; To make the folowing code simplier, we make sure that we have:
    ;   - a body #1 containing the colliding point
    ;   - a body #2 containing the colliding edge
    If VERLET_collisionInfo\ptrEdge\ptrParent = ptrBody2
      *VERLET_ptrCollPointBody = ptrBody1
      *VERLET_ptrCollEdgeBody = ptrBody2
    Else
      *VERLET_ptrCollPointBody = ptrBody2
      *VERLET_ptrCollEdgeBody = ptrBody1
    EndIf
    
    ; The collision normal must point at body #1
    VEC2_substract(*VERLET_ptrCollEdgeBody\center,*VERLET_ptrCollPointBody\center,VERLET_tempVector)
    ; if not, revert the collision normal
    If VEC2_dotproduct(VERLET_collisionInfo\normal,VERLET_tempVector) < 0
      VERLET_collisionInfo\normal\x = -VERLET_collisionInfo\normal\x
      VERLET_collisionInfo\normal\y = -VERLET_collisionInfo\normal\y
    EndIf
  
    ; The colliding vertex is the one closer to the center of body #2
    ; So we measure the distance of the vertex from the line using the line equation
    *VERLET_ptrGen = *VERLET_ptrCollPointBody\ptrPointList
    *VERLET_ptrPoint = PeekI(*VERLET_ptrGen)
    VEC2_substract(*VERLET_ptrCollEdgeBody\center,*VERLET_ptrPoint,VERLET_tempVector)
    VERLET_distance = VEC2_dotProduct(VERLET_collisionInfo\normal,VERLET_tempVector)
    VERLET_minDistance = VERLET_Distance
    VERLET_collisionInfo\ptrPoint = *VERLET_ptrPoint    
    For VERLET_i = 1 To *VERLET_ptrCollPointBody\nbPoint
      *VERLET_ptrGen + #SIZEOFPTR
      *VERLET_ptrPoint = PeekI(*VERLET_ptrGen)
      
      ; Measure the distance of the vertex from the line using the line equation
      VEC2_substract(*VERLET_ptrCollEdgeBody\center,*VERLET_ptrPoint,VERLET_tempVector)
      VERLET_distance = VEC2_dotProduct(VERLET_collisionInfo\normal,VERLET_tempVector)
      
      ; If the measured distance is smaller than the min distance, we've got our collision vertex
      If VERLET_distance < VERLET_minDistance
        VERLET_minDistance = VERLET_Distance
        VERLET_collisionInfo\ptrPoint = *VERLET_ptrPoint
      EndIf
    Next VERLET_i
    
  ;    Protected w.i,h.i
  ; ;  If (*VERLET_ptrCollPointBody\name = "bascule" And Left(*VERLET_ptrCollEdgeBody\name,6) = "Bridge") Or (*VERLET_ptrCollEdgeBody\name = "bascule" And Left(*VERLET_ptrCollPointBody\name,6) = "Bridge")
  ;     Repeat
  ;       ClearScreen($000001)
  ;       StartDrawing(ScreenOutput())
  ;       drawconstraints()
  ;       drawpointmasses()
  ;       Circle(VERLET_collisionInfo\ptrPoint\x,VERLET_collisionInfo\ptrPoint\y,2,$0000FF)
  ;       w = VERLET_collisionInfo\ptrEdge\p2\x - VERLET_collisionInfo\ptrEdge\p1\x
  ;       h = VERLET_collisionInfo\ptrEdge\p2\y - VERLET_collisionInfo\ptrEdge\p1\y
  ;       If w = 0
  ;         w = 1
  ;       EndIf
  ;       If h = 0
  ;         h = 1
  ;       EndIf
  ;       Line(VERLET_collisionInfo\ptrEdge\p1\x,VERLET_collisionInfo\ptrEdge\p1\y,w,h,$0000FF)
  ;       ;Line(*VERLET_ptrCollEdgeBody\center\x,*VERLET_ptrCollEdgeBody\center\y,VERLET_axis\x,VERLET_axis\y,$00FFFF)
  ;       StopDrawing()
  ;       FlipBuffers()
  ;       ExamineKeyboard()      
  ;     Until KeyboardReleased(#PB_Key_Space)
  ; ;   EndIf   
  
    ; Compute the collision response
    COLLISION_RESPONSE()

  EndIf ; if collisionResult = #true...
  
  
EndMacro

;************************************************************************************
; Name: UPDATE_POINTMASSES
; Purpose: Update position of each pointmass and add gravity (except for the static points)
; Parameters:
;   None
;************************************************************************************
Macro UPDATE_POINTMASSES()
  
  ForEach pointmass()
    If pointmass()\invmass <> 0
      
      ;gets the velocity  
		  VERLET_dx = pointmass()\x - pointmass()\oldX	
		  VERLET_dy = pointmass()\y - pointmass()\oldY + #GRAVITY 
		  
	    ;updates oldX and oldY
  		pointmass()\oldX = pointmass()\x
  		pointmass()\oldY = pointmass()\y
  		
  		;adds velocity to get new x and new y
  		pointmass()\x + VERLET_dx 	
  		pointmass()\y + VERLET_dy 
  		
  		If pointmass()\x > #SCREENWIDTH 
  			pointmass()\x = #SCREENWIDTH
  			VERLET_dy = pointmass()\y - pointmass()\oldY
  			pointmass()\oldY + VERLET_dy * #FRICTION	;simulates friction
  		EndIf
  		If pointmass()\x < 0 
  			pointmass()\x = 0
  			VERLET_dy = pointmass()\y - pointmass()\oldY
  			pointmass()\oldY + VERLET_dy * #FRICTION	;simulates friction
  		EndIf

  		If pointmass()\y > #SCREENHEIGHT 
  			pointmass()\y = #SCREENHEIGHT
  			VERLET_dx = pointmass()\x - pointmass()\oldX
  			pointmass()\oldX + VERLET_dx * #FRICTION	;simulates friction
  		EndIf
  		
  	EndIf
  Next pointmass()
  
EndMacro

;************************************************************************************
; Name: UPDATE_CONSTRAINTS
; Purpose: Once the points moved, the constraints do their best to retrieve their
;          initial length
; Parameters:
;   None
;************************************************************************************
Macro UPDATE_CONSTRAINTS()

  For VERLET_i = 1 To #CONSTRAINT_ITERATIONS	 ;this is necessary with many constraints to solve them correctly
    ForEach constraint()
	    
	    If constraint()\p1\invmass > 0 Or constraint()\p2\invmass > 0
  	    VERLET_dx = constraint()\p2\x - constraint()\p1\x
  	    VERLET_dy = constraint()\p2\y - constraint()\p1\y
  	    
  	    ; Correction to apply
   			;VERLET_deltalength = Sqr(VERLET_dx*VERLET_dx + VERLET_dy*VERLET_dy)	;distance formula
   			;VERLET_diff = (VERLET_deltalength - constraint()\length) / (VERLET_deltalength*(constraint()\p1\invmass+constraint()\p2\invmass)) 			
  
        ; Same thing, but without SQR (faster!)
        ;length2 = constraint()\length*constraint()\length ; optimisation: length2 is stored
        VERLET_diff = ( (constraint()\length2 / ((VERLET_dx*VERLET_dx + VERLET_dy*VERLET_dy) + constraint()\length2)) - 0.5) / ((constraint()\p1\invmass + constraint()\p2\invmass) * -0.5)
        
    	  constraint()\p1\x + (VERLET_diff * VERLET_dx * constraint()\p1\invmass)
    	  constraint()\p1\y + (VERLET_diff * VERLET_dy * constraint()\p1\invmass)
    	  constraint()\p2\x - (VERLET_diff * VERLET_dx * constraint()\p2\invmass)
    	  constraint()\p2\y - (VERLET_diff * VERLET_dy * constraint()\p2\invmass)
    	EndIf
    	
    Next constraint()
	Next VERLET_i	
	
EndMacro

;************************************************************************************
; Name: MANAGE_COLLISIONS
; Purpose: Check collision for each body against all the others (include collision
;          response => bounce)
; Parameters:
;   None
;************************************************************************************
Macro MANAGE_COLLISIONS()
  
    ; Compute center and "collision box" for each rigid body
    ForEach body()     
      *ptr = body()\ptrPointList
      *ptrPoint = PeekI(*ptr)
      body()\center\x = *ptrPoint\x
      body()\center\y = *ptrPoint\y
      body()\minX = *ptrPoint\x
      body()\maxX = *ptrPoint\x
      body()\minY = *ptrPoint\y
      body()\maxY = *ptrPoint\y
      
      For VERLET_i = 1 To body()\nbPoint
        *ptr + #SIZEOFPTR
        *ptrPoint = PeekI(*ptr)
        body()\center\x + *ptrPoint\x
        body()\center\y + *ptrPoint\y
        If *ptrPoint\x < body()\minX
          body()\minX = *ptrPoint\x
        EndIf
        If *ptrPoint\x > body()\maxX
          body()\maxX = *ptrPoint\x
        EndIf
        If *ptrPoint\y < body()\minY
          body()\minY = *ptrPoint\y
        EndIf
        If *ptrPoint\y > body()\maxY
          body()\maxY = *ptrPoint\y
        EndIf
      Next VERLET_i

      ; At this point, VERLET_i = body()\nbPoint + 1
      body()\center\x / VERLET_i
      body()\center\y / VERLET_i
    Next body()
    
    ; Collisions: Check each body against all the others
    ForEach body()
      *VERLET_prevBodyPtr = @body()
      VERLET_startIndex = ListIndex(body())
      ForEach body()
        If ListIndex(body()) > VERLET_startIndex And (*VERLET_prevBodyPtr\ptrParent = 0 Or body()\ptrParent = 0 Or *VERLET_prevBodyPtr\ptrParent <> body()\ptrParent)
          If ( *VERLET_prevBodyPtr\MinX <= body()\MaxX ) And ( *VERLET_prevBodyPtr\MinY <= body()\MaxY ) And ( *VERLET_prevBodyPtr\MaxX >= body()\MinX ) And ( *VERLET_prevBodyPtr\MaxY >= body()\MinY )
            DETECT_COLLISION(*VERLET_prevBodyPtr,body(),VERLET_isCollision)
          EndIf
        EndIf
      Next body()
      ChangeCurrentElement(body(),*VERLET_prevBodyPtr)
    Next body()
    
EndMacro    


;- Points / constraints creation

;************************************************************************************
; Name: createPointmass
; Purpose: Creates a pointmass
; Parameters:
;   - coords x,y (float)
;   - mass (optional float)
;   - initial velocities x,y (optional float)
; Returns : pointer to newly created pointmass
;************************************************************************************
Procedure createPointmass(x.f,y.f,mass.f = 1,vx.f = 0 ,vy.f = 0)	; x and y are coords for the verlet. vx and vy are velocity values for the verlet
  
  AddElement(pointmass())
  pointmass()\x = x
  pointmass()\y = y
  pointmass()\oldX = x - vx	;gives the particle a starting velocity
  pointmass()\oldY = y - vy
  If mass > 0
    pointmass()\mass = mass
    pointmass()\invmass = 1 / mass
  Else
    pointmass()\mass = 999999
    pointmass()\invmass = 0
  EndIf
  
  ProcedureReturn @pointmass()
EndProcedure


;************************************************************************************
; Name: createConstraint
; Purpose: Creates a constraint
; Parameters:
;   - pointer to first point (ptr)
;   - pointer to second point (ptr)
;   - is this constraint can be collided (optional boolean)
; Returns : pointer to newly created constraint
;************************************************************************************
Procedure createConstraint(*p1.pointmass_struct,*p2.pointmass_struct,enable.b = #True)
  
  AddElement(constraint())
	constraint()\p1 = *p1
	constraint()\p2 = *p2
	constraint()\enable = enable
	constraint()\length = Sqr( Pow((constraint()\p1\x - constraint()\p2\x),2) + Pow((constraint()\p1\y - constraint()\p2\y),2) )
	constraint()\length2 = constraint()\length * constraint()\length ; square length. Stored for optimization
	
	ProcedureReturn @constraint()

EndProcedure


;************************************************************************************
; Name: createBody
; Purpose: Creates a rigid body
; Parameters:
;   None (points and constraints will be added later)
; Returns : pointer to newly created body
;************************************************************************************
Procedure createBody()
  VERLET_numBody + 1
  
  AddElement(body())
  body()\numint = VERLET_numBody
  body()\nbPoint = -1
  body()\nbEdge = -1
  
  ProcedureReturn @body()  
EndProcedure

;************************************************************************************
; Name: addBodyPointmass
; Purpose: Creates a pointmass and adds it to an already created body
; Parameters:
;   - pointer to the body (ptr)
;   - coords x,y (float)
;   - mass (optional float)
;   - initial velocities x,y (optional float)
; Returns : pointer to newly created pointmass
;************************************************************************************
Procedure addBodyPointmass(*ptrBody.rigidBody_struct ,x.f,y.f,mass.f = 1,vx.f = 0 ,vy.f = 0)
  Protected *ptr.pointmass_struct
  
  ; Make room for the new point in the body's list
  *ptrBody\nbPoint + 1
  If *ptrBody\nbPoint = 0
    *ptrBody\ptrPointList = AllocateMemory((*ptrBody\nbPoint + 1) * #SIZEOFPTR )
  Else
    *ptrBody\ptrPointList = ReAllocateMemory(*ptrBody\ptrPointList,(*ptrBody\nbPoint + 1) * #SIZEOFPTR )
  EndIf
  
  ; Create the point and store its pointer in the list
  *ptr = createPointmass(x,y,mass,vx,vy)
  PokeI(*ptrBody\ptrPointList + *ptrBody\nbPoint * #SIZEOFPTR , *ptr )
  
  ProcedureReturn *ptr
EndProcedure


;************************************************************************************
; Name: addBodyConstraint
; Purpose: Creates a constraint and adds it to an already created body
; Parameters:
;   - pointer to the body (ptr)
;   - pointer to first point (ptr)
;   - pointer to second point (ptr)
;   - is this constraint can be collided (optional boolean)
; Returns : pointer to newly created constraint
;************************************************************************************
Procedure addBodyConstraint(*ptrBody.rigidBody_struct,*p1.pointmass_struct,*p2.pointmass_struct,enable.b = #True)
  Protected *ptr.constraint_struct
  
  ; Make room for the new constraint in the body's list
  *ptrBody\nbEdge + 1
  If *ptrBody\nbEdge = 0
    *ptrBody\ptrConstraintList = AllocateMemory((*ptrBody\nbEdge + 1) * #SIZEOFPTR )
  Else
    *ptrBody\ptrConstraintList = ReAllocateMemory(*ptrBody\ptrConstraintList,(*ptrBody\nbEdge + 1) * #SIZEOFPTR )
  EndIf
      
  ; Create the constraint and store its pointer in the list
  *ptr = createConstraint(*p1,*p2,enable)
  PokeI(*ptrBody\ptrConstraintList + *ptrBody\nbEdge * #SIZEOFPTR, *ptr)
  *ptr\ptrParent = *ptrBody
   
  ProcedureReturn *ptr
EndProcedure


; IDE Options = PureBasic 4.51 (Windows - x86)
; CursorPosition = 16
; Folding = ---
; EnableXP