<?php

// ZIM_Base() - class to help with Database actions
// See https://codepen.io/danzen/pen/oNjgMNX for front end example
// See provided hero.php for backend example
// See https://zimjs.com/docs.html?item=Bind for Bind docs in ZIM
// See https://zimjs.com/skool/lesson09.html for Data Theory and explanation
// See https://zimjs.com/learnjavascript.html#lesson09 for video support

class ZIM_Base {

    public $mysqli;
    public $type; // these are also available as globals if globals is true
    public $master;
    public $data;
    public $command;
    public $extra;
    public $lock;
    public $unique;

    public function __construct($mysqli, $globals=true) {
        $this->mysqli = $mysqli;
        if ($globals) {
            global $type, $master, $data, $command, $extra, $lock, $unique;
        }

        // Declare the type of content we will be outputting
        // This is needed for JSONp with GET - harmless with POST
        // PHP has a variety of headers for mail, images, etc.
        // Usually we do not send headers. If you do, they must be output first
        header('Content-type: text/javascript');

        // COLLECTING DATA
        // We have arranged our ZIM Bind code to send the type of connection
        // This is placed on the URL so it gets collected with with GET regardless of type
        // The value will be either "get" or "post" (note: GET=="get" and POST=="post" in ZIM)
        // The rest of the code handles changing between the two automatically

        $this->type = $type = isset($_GET["type"]) ? $_GET["type"] : "get";

        // We collect get or post data as $_GET["property"] or $_POST["property"]
        // These are global environment variables in PHP
        // Data can come from an HTML form, JSONp call, AJAX call, etc. it does not matter
        // We use the PHP isset() function to find out if the property exists
        // and if not, then we set a default - here we set the default "get"
        // This code is the Ternary operator common in most languages
        // It is like an if statement all in one line
        // if there is a type then assign the type to $type else assign "get" to $type

        // ZIM Bind methods send the following variables
        // We may not use them all but it is handy to just copy this full code block
            // master - information for all "to" and "from" - like a room, etc.
            // data - the JSON data to store or the data requested
            // command - "to" or "from" to store or retrieve data
            // extra - holds extra information we might need like an id
            // lock - are we wanting to record lock - see ZIM PHP setLock() function
            // unique - is whether we want our "to" to be unique (if not, we can update)

        $inputs = ["master", "data", "command", "extra", "lock", "unique"];
        foreach($inputs as $input) {
            if ($this->type == "get") {
                $this->$input = $$input = isset($_GET[$input]) ? $_GET[$input] : "";
            } else {
                $this->$input = $$input = isset($_POST[$input]) ? $_POST[$input] : "";
            }
        }
    }

    // ~~~~~~~~~~~~~~
    // PRIVATE FUNCTIONS

    private function getWhere($where) {
        $names = [];
        $vals = [];
        $types = [];
        if (isset($where)) {
            $string = " WHERE ";
            $count = 0;
            foreach ($where as $n=>$v) {
                if (gettype($v)=="array") {
                    // [prefix, infix, value]
                    $pre = " $v[0] ";
                    if (!$v[1] || $v[1]=="") $v[1]="=";
                    array_push($names, "$n".$v[1]."?");
                    array_push($vals, $v[2]);
                } else {
                    $pre = $count==0?"":" AND ";
                    array_push($names, "$n=?");
                    array_push($vals, $v);
                }
                array_push($types, gettype($v)=="integer"?"i":"s");
                $string .= $pre . $names[$count];
                $count++;
            }
        } else {
            $string = "";
        }
        return [$names, $vals, $types, $string];
    }

    private function getField($fields) {
        $names = [];
        $binds = [];
        $vals = [];
        $types = [];
        foreach ($fields as $n=>$v) {
            array_push($names, $n);
            if ($v=="NOW()") {
                array_push($binds, $v);
            } else if ($v==null) {
                array_push($binds, "NULL");
            } else {
                array_push($binds, "?");
                array_push($vals, $v);
                array_push($types, gettype($v)=="integer"?"i":"s");
            }
        }
        $string = "(".join($names,", ").") VALUES (".join($binds,", ").")";
        return [$names, $vals, $types, $string];
    }

    private function getUpdate($update) {
        $names = [];
        $binds = [];
        $vals = [];
        $types = [];
        $pairs = [];
        $count = 0;
        foreach ($update as $n=>$v) {
            array_push($names, $n);
            if ($v=="NOW()") {
                array_push($binds, $v);
            } else if ($v==null) {
                array_push($binds, "NULL");
            } else {
                array_push($binds, "?");
                array_push($vals, $v);
                array_push($types, gettype($v)=="integer"?"i":"s");
            }
            array_push($pairs, $n."=".$binds[$count]);
            $count++;
        }
        $string = "UPDATE " . join($pairs,", ");
        return [$names, $vals, $types, $string];
    }

    // ~~~~~~~~~~~~~~
    // PUBLIC FUNCTIONS

    // select(table, fields, where, more)
    // select(string, string or array, assoc with value or [prefix, infix, value], string)
    // EXAMPLES
    // $response = $base->select("zim_hero", "name", [name=>$name]);
    // $response = $base->select("zim_hero", ["name","age"], [name=>$name, age=>$age]);
    // $response = $data->select("zim_hero", ["name","age"], [name=>["","!=",$name], age=>["OR",null,$age]], "ORDER BY count DESC LIMIT 10");
    // RESPONSE OBJECT METHODS AND PROPERTIES
    // $response->query - the SQL statement
    // $response->error - true or false
    // $response->num_rows - number of rows selected
    // $response->result - array of associative arrays matching selected fields eg. $response->result[1]["data"]
    // $response->array - array of arrays in order of matching fields eg. $response->result[1][0]
    // $response->json - json array of former json objects - assumes one json field was selected - decodes, adds to array and encodes
    public function select($table, $fields="*", $where=null, $more=null) {

        if ($fields == "*") {
            $fileString = "*";
        } else if (gettype($fields) == "array") {
            $fieldString = join($fields, ", ");
        } else {
            $fieldString = $fields;
        }
        list($names, $vals, $types, $whereString) = $this->getWhere($where);
        $more = isset($more)?$more:"";
        $query = "SELECT " . $fieldString . " FROM " . $table . $whereString . " " . $more;

        $obj = new stdClass();
        $obj->query = $query;

        $stmt =  $this->mysqli->stmt_init();
        if ($stmt->prepare($query)) {
            $stmt->bind_param(join($types,""), ...$vals);
            $stmt->execute();
            $result = $stmt->get_result();
            $stmt->close();
            $obj->num_rows = $result->num_rows;
            $obj->error = false;
            $obj->result = $result->fetch_all(MYSQLI_ASSOC);
            $result->data_seek(0);
            $obj->array = $result->fetch_all(MYSQLI_NUM);
            $arranged = [];
            foreach($obj->array as $d) {
                array_push($arranged, json_decode($d[0]));
            }
            $obj->json = json_encode($arranged);
        } else {
            $obj->error = true;
        }
        return $obj;
    } // end select

    // insert(table, fields, update, where, more)
    // insert(string, assoc array, assoc array, assoc with value or [prefix, infix, value], string)
    // EXAMPLES
    // $response = $base->insert("zim_hero", [name=>$name, data=>$data, date=>"NOW()"], [data=>$data]);
    // see insert() for example of $where and $more
    // RESPONSE OBJECT METHODS AND PROPERTIES
    // $response->query - the SQL statement
    // $response->error - true or false

    public function insert($table, $fields, $update=null, $where=null, $more=null) {

        // $query = "INSERT INTO zim_hero (name, data, date) VALUES (?, ?, NOW()) ON DUPLICATE KEY UPDATE data = ?, name =?";

        list($fieldNames, $fieldVals, $fieldTypes, $fieldString) = $this->getField($fields);
        list($whereNames, $whereVals, $whereTypes, $whereString) = $this->getWhere($where);
        $more = isset($more)?$more:"";
        list($updateNames, $updateVals, $updateTypes, $updateString) = $this->getUpdate($update);

        $updateString = " ON DUPLICATE KEY " . $updateString;
        $query = "INSERT INTO " . $table ." ". $fieldString ." ". $whereString ." ". $more ." ". $updateString;

        $types = array_merge($fieldTypes, $whereTypes, $updateTypes);
        $vals = array_merge($fieldVals, $whereVals, $updateVals);

        $obj = new stdClass();
        $obj->query = $query;

        $stmt = $this->mysqli->stmt_init();
        if ($stmt->prepare($query)) {
            $stmt->bind_param(join($types,""), ...$vals);
            $stmt->execute();
            $stmt->close();
            $obj->error = false;
        } else {
            $obj->error = true;
        }
        return $obj;
    } // end insert

    // delete(table, where)
    // delete(string, array)
    // EXAMPLES
    // $response = $base->delete("zim_hero", [name=>$name]);
    // $response = $data->delete("zim_hero", [name=>["NOT","=",$name]]);
    // RESPONSE OBJECT METHODS AND PROPERTIES
    // $response->query - the SQL statement
    // $response->affected_rows - the number of rows deleted
    // $response->error - true or false
    public function delete($table, $where=null) {

        list($names, $vals, $types, $whereString) = $this->getWhere($where);
        $query = "DELETE FROM " . $table . $whereString;

        $obj = new stdClass();
        $obj->query = $query;

        $stmt =  $this->mysqli->stmt_init();
        if ($stmt->prepare($query)) {
            $stmt->bind_param(join($types,""), ...$vals);
            $stmt->execute();
            $obj->affected_rows = $stmt->affected_rows;
            $stmt->close();
            $obj->error = false;
        } else {
            $obj->error = true;
        }
        return $obj;
    } // end delete


    // reply(messageType, message)
    // reply(string, string)
    // EXAMPLES
    // if ($name == "") $base->reply("error","missing id");
    // or
    // $base->reply($response->json);
    public function reply($messageType, $message=null) {
        $type = $this->type; // do not rely on globals as they could be turned off
        $command = $this->command;
        // for JSONp, each command has a different callback function
        // we could dynamically create the function names but we will keep it simple
        if ($message) {
            $message = json_encode([$messageType=>$message]);
        } else {
            $message = $messageType;
        }
        if ($command == "to") echo $type == "get" ? "async.callbackTo($message)" : $message;
        else if ($command == "fromTo") echo $type == "get" ? "async.callbackFromTo($message)" : $message;
        else if ($command == "from") echo $type == "get" ? "async.callbackFrom($message)" : $message;
        else if ($command == "removeAll") echo $type == "get" ? "async.callbackRemoveAll($message)" : $message;
        exit; // should only ever send one message back
    }
}


?>
