import { z } from "zod";

import type { InferContract, InferContractParams, InferContractRequest } from "@devowl-wp/api";
import {
    ContractResponse,
    EHttpStatusCodeClientError,
    EHttpStatusCodeSuccess,
    ERouteHttpVerb,
    createContract,
} from "@devowl-wp/api";

import { createContractGuardJwt } from "../../guard/jwt.js";

const createContractUntranslatableDelete = () =>
    createContract({
        z,
        versions: ["1.0.0"],
        route: {
            path: "/template/untranslatable",
            method: ERouteHttpVerb.DELETE,
        },
        routeDetails: {
            shortDescription: "Unlink an untranslatable string",
            longDescription:
                "Unlink an untranslatable string from a service template or content blocker. If an untranslatable string has no more references, it will be finally deleted.",
            groups: ["Untranslatable"],
        },
        guards: () => ({
            jwt: createContractGuardJwt(),
        }),
        request: () =>
            z.object({
                text: z.string(),
                template: z.object({
                    type: z.enum(["Service", "Content-Blocker"]),
                    identifier: z.string(),
                }),
            }),
        response: () => ({
            [EHttpStatusCodeSuccess.NoContent]: new ContractResponse<void>(),
            [EHttpStatusCodeClientError.NotFound]: new ContractResponse<void>(),
        }),
    });

type UntranslatableDeleteContract = InferContract<typeof createContractUntranslatableDelete>;

interface IUntranslatableDeleteContractRequest
    extends InferContractRequest<typeof createContractUntranslatableDelete> {}
interface IUntranslatableDeleteContractParams extends InferContractParams<typeof createContractUntranslatableDelete> {}
interface IUntranslatableDeleteContractResponse {}

export {
    type IUntranslatableDeleteContractParams,
    type IUntranslatableDeleteContractRequest,
    type IUntranslatableDeleteContractResponse,
    type UntranslatableDeleteContract,
    createContractUntranslatableDelete,
};
