import type { Declarations, IntMap, Vendor, VendorList } from "@iabtechlabtcf/core";

// Instead of relying on the `@iabtechlabtcf/core` package, we use our own types
// as we made the experience that TCF does not update the package regularly and keep the
// types streamlined with the documentation / GVL specification.

enum ETcfGvlVersion {
    v2 = "v2",
    v3 = "v3",
}

enum ETcfGvlDisclosureType {
    Cookie = "cookie",
    Web = "web",
    App = "app",
}

/**
 * @see https://github.com/InteractiveAdvertisingBureau/GDPR-Transparency-and-Consent-Framework/blob/master/TCFv2/Vendor%20Device%20Storage%20%26%20Operational%20Disclosures.md#disclosures-array
 */
interface ITcfGvlDisclosure {
    identifier: string;
    type: ETcfGvlDisclosureType;
    maxAgeSeconds?: number;
    cookieRefresh?: boolean;
    domain?: string;
    domains?: string[];
    purposes: number[];
    specialPurposes?: number[];
    description?: string;
    optOut?: boolean;
}

/**
 * @see https://github.com/InteractiveAdvertisingBureau/GDPR-Transparency-and-Consent-Framework/blob/master/TCFv2/Vendor%20Device%20Storage%20%26%20Operational%20Disclosures.md#domains-array
 */
interface ITcfGvlDisclosureDomain {
    domain: string;
    use?: string;
}

/**
 * @see https://github.com/InteractiveAdvertisingBureau/GDPR-Transparency-and-Consent-Framework/blob/master/TCFv2/Vendor%20Device%20Storage%20%26%20Operational%20Disclosures.md#sdks-array
 */
interface ITcfGvlSdk {
    name: string;
    use?: string;
}

interface ITcfGvlDeviceStorageDisclosure {
    disclosures: ITcfGvlDisclosure[];
    domains?: ITcfGvlDisclosureDomain[];
    sdks?: ITcfGvlSdk[];
}

type ITcfGvlAdditionalInformationTerritorialScope =
    | "BE"
    | "BG"
    | "CZ"
    | "DK"
    | "DE"
    | "EE"
    | "IE"
    | "GR"
    | "ES"
    | "FR"
    | "HR"
    | "IS"
    | "IT"
    | "CY"
    | "LV"
    | "LI"
    | "LT"
    | "LU"
    | "HU"
    | "MT"
    | "NL"
    | "NO"
    | "AT"
    | "PL"
    | "PT"
    | "RO"
    | "SI"
    | "SK"
    | "FI"
    | "SE"
    | "CH"
    | "GB";
type ITcfGvlAdditionalInformationEnvironment = "web" | "native app (mobile)" | "native app (CTV)";
type ITcfGvlAdditionalInformationServiceType =
    | "SSP"
    | "DSP"
    | "Verification"
    | "Ad serving"
    | "Header bidding"
    | "DMP / Data provider"
    | "Identity resolution services"
    | "Content delivery network"
    | "Recommendation service"
    | "Website analytics"
    | "Buyer campaign analytics"
    | "Audience analytics"
    | "Other";

/**
 * @see https://github.com/IABTechLab/fideslang/blob/main/src/fideslang/models.py#L208
 */
type ITcfGvlAdditionalInformationTransferMechanisms = "Adequacy decision" | "SCCs" | "BCRs" | "Other";

interface ITcfGvlAdditionalInformation {
    name: string;
    legalAddress: string;
    contact: string;
    territorialScope: ITcfGvlAdditionalInformationTerritorialScope[];
    environments: ITcfGvlAdditionalInformationEnvironment[];
    serviceTypes: ITcfGvlAdditionalInformationServiceType[];
    internationalTransfers: boolean;
    transferMechanisms?: ITcfGvlAdditionalInformationTransferMechanisms[];
}

interface IVendorExtended extends Vendor {
    deviceStorageDisclosure?: ITcfGvlDeviceStorageDisclosure;
    deviceStorageDisclosureViolation?:
        | "no-disclosures"
        | "disclosure-no-domains"
        | "disclosure-no-purposes"
        | "disclosure-no-max-age-seconds"
        | "disclosure-no-cookie-refresh"
        | "disclosure-too-big"
        | "unknown";
    additionalInformation?: ITcfGvlAdditionalInformation;
}

interface ITcfGvlVendorListComposed extends VendorList {
    vendors: IntMap<IVendorExtended>;
}

interface ITcfGvlPurposes extends Declarations {}

enum ETcfGvlLanguage {
    Bulgarian = "bg",
    Catalan = "ca",
    Czech = "cs",
    Danish = "da",
    German = "de",
    Greek = "el",
    Spanish = "es",
    Estonian = "et",
    Basque = "eus",
    Finnish = "fi",
    French = "fr",
    Galician = "gl",
    Croatian = "hr",
    Hungarian = "hu",
    Italian = "it",
    Japanese = "ja",
    Lithuanian = "lt",
    Latvian = "lv",
    Maltese = "mt",
    Dutch = "nl",
    Norwegian = "no",
    Polish = "pl",
    Portuguese = "pt",
    Romanian = "ro",
    SerbianCyrillic = "sr-Cyrl",
    SerbianLatin = "sr-Latn",
    Russian = "ru",
    Slovak = "sk",
    Slovenian = "sl",
    Swedish = "sv",
    Turkish = "tr",
    Chinese = "zh",
}

export {
    ETcfGvlVersion,
    type ITcfGvlVendorListComposed,
    type ITcfGvlPurposes,
    ETcfGvlLanguage,
    type IVendorExtended,
    type ITcfGvlDisclosure,
    ETcfGvlDisclosureType,
    type ITcfGvlDeviceStorageDisclosure,
    type ITcfGvlSdk,
    type ITcfGvlDisclosureDomain,
    type ITcfGvlAdditionalInformation,
    type ITcfGvlAdditionalInformationTerritorialScope,
    type ITcfGvlAdditionalInformationEnvironment,
    type ITcfGvlAdditionalInformationServiceType,
    type ITcfGvlAdditionalInformationTransferMechanisms,
};
