import type { ITranslatableProperties, ITranslationStatus } from "./translatable-properties.js";

/**
 * Used to mark an ORM Entity as "translated" object.
 * The generic holds all translatable properties for the object.
 *
 * @interface ITranslated
 * @template T
 */
interface ITranslated<T extends ITranslatableProperties> {
    id: string;
    translations?: T[];
    translationIds?: string[];
    translationInfo?: ITranslationStatus[];
}

/**
 * An object with information about the completeness of the translation, including whether
 * it's a translatable entity, has a complete translation, the list of translatable properties,
 * and information about the found translation.
 */
type TranslationCompleteCheckResults = {
    isTranslatableEntity: boolean;
    hasCompleteTranslation: boolean;
    translatableProperties: any[];
    foundTranslation: ITranslatableProperties;
};

/**
 * Translation flags for "tagging" translation string.
 * @see https://app.clickup.com/t/861n9jg7k (CU-861n9jg7k)
 */
enum ETranslationFlag {
    Fuzzy = "fuzzy",
    MachineTranslated = "machineTranslated",
}

/**
 * Represents a mapping of translation flags to arrays of field names.
 * Each key in the object corresponds to a translation flag (e.g., 'fuzzy', 'machineTranslated'),
 * and its value is an array of field names that have been flagged with that particular translation flag.
 *
 * This structure allows for efficient tracking and management of fields that have specific
 * translation statuses or require special attention during the translation process.
 *
 * @example
 * {
 *   fuzzy: ['title', 'description'],
 *   machineTranslated: ['content']
 * }
 */
type TranslationFlaggedFields = {
    [key in keyof typeof ETranslationFlag]: string[];
};

/**
 * Property name for the translations array.
 */
const TRANSLATIONS_PROPERTY_NAME = "translations";

/**
 * Property name for the translation info array.
 */
const TRANSLATION_INFO_PROPERTY_NAME = "translationInfo";

/**
 * Property name for the translation ids array.
 */
const TRANSLATION_IDS_PROPERTY_NAME = "translationIds";

/**
 * Suffix added to property names to indicate translation flags.
 *
 * @example
 * // For a property named 'title'
 * titleTranslationFlags: ETranslationFlag[];
 */
const TRANSLATION_FLAG_SUFFIX = "TranslationFlags";

export {
    ETranslationFlag,
    type ITranslated,
    type TranslationCompleteCheckResults,
    type TranslationFlaggedFields,
    TRANSLATIONS_PROPERTY_NAME,
    TRANSLATION_IDS_PROPERTY_NAME,
    TRANSLATION_INFO_PROPERTY_NAME,
    TRANSLATION_FLAG_SUFFIX,
};
