import type { BrokerAckPolicy, BrokerDeliverPolicy, BrokerRetention } from "./contract.js";
import type {
    BrokerEffectivelyOnceGuarantee,
    BrokerEffectivelyOnceNoGuarantee,
    BrokerEffectivelyOnceNone,
} from "./effectively-once.js";

/**
 * Broadcast without any guarantees that the message will be delivered to all subscribers.
 *
 * - Volatile, `at-most-once` semantic (no redeliveries, but message loss possible).
 * - No broker deduplication.
 * - Effectively Once optional (Publisher + Subscriber) through message ID
 *
 * Example use cases:
 * - Real-time metrics and telemetry data where occasional loss is acceptable
 * - System health pings and heartbeat signals
 * - Progress bar updates
 *
 * Example: Use NATS Core instead of JetStream (`at-most-once`).
 */
type BrokerBroadcastNoGuarantee = {
    type: "broadcast+no-guarantee";
} & (BrokerEffectivelyOnceNone | BrokerEffectivelyOnceNoGuarantee);

/**
 * Broadcast with a guarantee that the message will be delivered at least once to all subscribers.
 *
 * - Persistence/Replay possible, `at-least-once` semantic.
 * - Dedupe + double confirmation + Subscriber-Idempotency are enforced via "effectively once".
 *
 * Example use cases:
 * - System-wide announcements (maintenance notifications sent to all connected clients)
 * - Cache invalidation events (all application instances clear the same cached data)
 *
 * Example: Use JetStream instead of NATS Core (`at-least-once`).
 */
type BrokerBroadcastGuarantee = {
    type: "broadcast+guarantee";
    /**
     * Name of the stream to publish the message to (bridge between publisher and subscriber to be able
     * to replay the message and hold it for a while).
     *
     * @default Automatically generated
     */
    stream?: string;
    /**
     * Max age of the message in the stream in milliseconds. Afterward, it is removed from the stream.
     *
     * @default 60 * 60 * 1000 // 1 hour
     */
    ttlMsOnStream?: number;
    /**
     * Name of the durable subscription to use for the stream.
     *
     * Use `{durable}` as placeholder for the durable name (the name of the durable subscription).
     *
     * @default Automatically generated
     */
    durable?: string;
} & (BrokerRetention & {
    // `workqueue` is not supported for broadcast
    retention?: "limits" | "interest";
}) &
    BrokerDeliverPolicy &
    BrokerAckPolicy &
    BrokerEffectivelyOnceGuarantee;

export { type BrokerBroadcastNoGuarantee, type BrokerBroadcastGuarantee };
