import type { BrokerJsonSchema } from "./contract.js";
import type { BrokerEffectivelyOnceNoGuarantee, BrokerEffectivelyOnceNone } from "./effectively-once.js";

/**
 * Create a response from and for a broker contract.
 *
 * This is morely a workaround as TypeScript does currently not support partial generic types with infer-support.
 *
 * @see https://github.com/microsoft/TypeScript/issues/26242
 */
class BrokerContractResponse<Response> {
    public readonly response: Response;
}

/**
 * Request-Reply means that the publisher sends a message and the subscriber sends a response.
 *
 * For usual, we should try to use a REST API here but sometimes it is not possible or not practical
 * to introduce a REST API needed for exactly one use case.
 *
 * - No delivery semantics specification (implicitly `at-most-once`).
 * - Timeout is mandatory (client-side).
 *
 * Example use cases:
 * - User authentication - Validate login credentials and receive user profile data
 * - Order validation - Check inventory availability before confirming order placement
 *
 * Example: Use NATS Core instead of JetStream (`at-most-once`).
 */
type BrokerRequestReply = {
    type: "request+reply";
    /**
     * Timeout in milliseconds for the maximum time to wait for a response.
     *
     * @default 2_000 // 2 seconds
     */
    timeoutMs?: number;
    /**
     * JSON Schema of the response.
     */
    response: BrokerContractResponse<BrokerJsonSchema>;
} & (BrokerEffectivelyOnceNone | BrokerEffectivelyOnceNoGuarantee);

export { BrokerContractResponse, type BrokerRequestReply };
