import { EditOutlined, PlusOutlined } from "@ant-design/icons";
import { Avatar, Col, Row, Spin } from "antd";
import { useCallback, useState } from "react";

import type { ComponentProps, FC } from "react";

type ListMultilingualLanguagesRecords = Array<{
    code: string;
    language: string;
    flag: string | false;
    id: number | false;
    taxonomies?: Record<string, number[]>;
}>;

const ListMultilingualLanguages: FC<
    {
        languages: ListMultilingualLanguagesRecords;
        recordId: number;
        onClick: (recordId: number, translation: ListMultilingualLanguagesRecords[0]) => Promise<void>;
        wrapperProps?: Omit<ComponentProps<typeof Spin>, "spinning">;
    } & Omit<ComponentProps<typeof Row>, "align" | "onClick">
> = ({ languages = [], recordId, onClick, children, wrapperProps = {}, ...rest }) => {
    const [isBusy, setBusy] = useState(false);
    const createdTranslations = languages.filter(({ id }) => id !== false);
    const missingTranslations = languages.filter(({ id }) => id === false);

    const avatarAttrs: ComponentProps<typeof Avatar> = {
        size: 13,
        shape: "square",
        style: { display: "block", width: "auto", borderRadius: 5 },
    };

    const colAttrs: ComponentProps<typeof Col> = {
        flex: "none",
        style: { textAlign: "center", paddingRight: 5, fontSize: 11, cursor: "pointer" },
    };

    const handleClick = useCallback(
        async (translation: ListMultilingualLanguagesRecords[0]) => {
            if (!isBusy) {
                setBusy(true);
                try {
                    onClick(recordId, translation);
                } finally {
                    setBusy(false);
                }
            }
        },
        [isBusy, onClick, recordId],
    );

    return (
        <Spin spinning={isBusy} {...wrapperProps}>
            <Row align="middle" {...rest}>
                {createdTranslations.map((translation) => {
                    const { code, flag, id } = translation;
                    return id === recordId ? null : (
                        <Col key={code} {...colAttrs} onClick={() => handleClick(translation)}>
                            <Avatar src={flag} {...avatarAttrs}>
                                {code.toUpperCase()}
                            </Avatar>
                            <EditOutlined />
                        </Col>
                    );
                })}
                {missingTranslations.length > 0 &&
                    missingTranslations.map((translation) => {
                        const { code, flag } = translation;
                        return (
                            <Col key={code} {...colAttrs} onClick={() => handleClick(translation)}>
                                <Avatar src={flag} {...avatarAttrs}>
                                    {code.toUpperCase()}
                                </Avatar>
                                <PlusOutlined />
                            </Col>
                        );
                    })}
                {children}
            </Row>
        </Spin>
    );
};

export { ListMultilingualLanguages, type ListMultilingualLanguagesRecords };
