import { Form, Input } from "antd";
import { useState } from "react";

import { useI18n } from "../../../../../contexts/i18n.js";
import { useCookieExpertsModal } from "../../../../../hooks/useCookieExpertsModal.js";
import { LearnMoreTag } from "../../../../common/learnMoreTag.js";
import { FormValueDifferFromTemplateTag } from "../../../valueDifferFromTemplateTag.js";

import type { FormContentBlockerValueProps } from "../../../../../types/formContentBlocker.js";
import type { FC } from "react";

const FormContentBlockerFieldTechnicalRules: FC = () => {
    const { __, _i } = useI18n();
    const { openDialog: openCookieExperts } = useCookieExpertsModal();
    const [ref, setRef] = useState<HTMLParagraphElement>();

    return (
        <Form.Item<FormContentBlockerValueProps> label={__("URLs / Elements to block")} required>
            <Form.Item
                name="rules"
                noStyle
                rules={[{ required: true, message: __("Please provide at least one URL/element!") }]}
            >
                <Input.TextArea autoSize={{ minRows: 3, maxRows: 15 }} />
            </Form.Item>
            <p className="description" ref={setRef}>
                <FormValueDifferFromTemplateTag
                    form="blocker"
                    valueName="rules"
                    widthOfRef={ref}
                    difference={(oldValue: string, newValue: string) => {
                        const currentRules = oldValue.split("\n");
                        const diff = newValue.split("\n").filter((newRule) => !currentRules.includes(newRule));
                        return diff.length > 0 ? diff : undefined;
                    }}
                    apply={(diff, setFieldsValue, oldValue) =>
                        setFieldsValue({
                            rules: oldValue.split("\n").concat(diff).join("\n"),
                        })
                    }
                    newValueText={__("Missing entries:")}
                    renderDiff={(value, diff) => (
                        <Input.TextArea value={diff.join("\n")} readOnly autoSize={{ minRows: 3, maxRows: 15 }} />
                    )}
                />
                {_i(
                    __(
                        "Enter one rule per line to replace content with a content blocker. You can block all available URLs or HTML tags on your website including videos, iframes, scripts, inline scripts and stylesheets. Please use an asterisk ({{code}}*{{/code}}) as a wildcard (placeholder).",
                    ),
                    { code: <code /> },
                )}
                &nbsp;&bull;&nbsp;
                {!!openCookieExperts && (
                    <button type="button" className="button-link" onClick={openCookieExperts}>
                        {__("Can't handle it? Let a Cookie Expert help you!")}
                    </button>
                )}
                <br />
                <br />
                {_i(
                    __(
                        '{{strong}}Pro tip:{{/strong}} Look up all available syntaxes like {{code}}div[class*="my-embed"]{{/code}} in our knowledge base to block content perfectly.',
                    ),
                    { strong: <strong />, code: <code /> },
                )}
                {` `}
                <LearnMoreTag
                    url={__("https://devowl.io/knowledge-base/real-cookie-banner-create-individual-content-blocker/")}
                />
            </p>
        </Form.Item>
    );
};

export { FormContentBlockerFieldTechnicalRules };
