import { Card, Form, Radio } from "antd";
import { useMemo } from "react";

import { useI18n } from "../../../../../contexts/i18n.js";
import { useUpsell } from "../../../../../contexts/upsell.js";
import { useUpsellModalVisualContentBlocker } from "../../../../../hooks/useUpsellModalVisualContentBlocker.js";
import { FormContentBlockerLayout } from "../../../layout.js";

import type { FormContentBlockerValueProps } from "../../../../../types/formContentBlocker.js";
import type { FC } from "react";

const FormContentBlockerFieldVisualType: FC = () => {
    const { __ } = useI18n();
    const { isPro } = useUpsell();
    const { modal, tag } = useUpsellModalVisualContentBlocker();

    const visualTypes = useMemo(
        () => ({
            default: {
                title: __("Textbox"),
                description: __("Plain text with button"),
            },
            wrapped: {
                title: __("Wrapped"),
                description: __("Image surrounding the content blocker"),
            },
            hero: {
                title: __("Hero"),
                description: __("Image with content blocker on click"),
            },
        }),
        [__],
    );

    return (
        <Form.Item<FormContentBlockerValueProps>
            noStyle
            shouldUpdate={(prevValues, nextValues) => prevValues.isVisual !== nextValues.isVisual}
        >
            {({ getFieldValue }) =>
                !!getFieldValue("isVisual") && (
                    <Form.Item<FormContentBlockerValueProps>
                        wrapperCol={{ offset: FormContentBlockerLayout.labelCol.span }}
                        style={{ paddingBottom: 10 }}
                    >
                        <span>
                            <Form.Item<FormContentBlockerValueProps>
                                name="visualType"
                                noStyle
                                rules={
                                    isPro
                                        ? []
                                        : [
                                              {
                                                  required: true,
                                                  type: "enum",
                                                  enum: ["default"],
                                                  message: __(
                                                      "This type of visual content blocker is available only in the PRO version of the plugin. Please choose a textbox content blocker!",
                                                  ),
                                              },
                                          ]
                                }
                            >
                                <Radio.Group
                                    size="large"
                                    buttonStyle="solid"
                                    className="rcb-antd-radio-button-card"
                                    style={{ marginBottom: 10 }}
                                >
                                    {Object.keys(visualTypes).map((type) => {
                                        const { description, title } = visualTypes[type as keyof typeof visualTypes];
                                        return (
                                            <Radio.Button value={type} key={type}>
                                                <Card
                                                    style={{ width: 300 }}
                                                    bordered={false}
                                                    cover={
                                                        <img
                                                            style={{ height: 168.75 }}
                                                            src={`https://assets.devowl.io/in-app/wp-real-cookie-banner/visual-content-blocker-preview/${type}.png`}
                                                        />
                                                    }
                                                >
                                                    <Card.Meta
                                                        title={
                                                            <>
                                                                {modal}
                                                                {title} {type !== "default" && tag}
                                                            </>
                                                        }
                                                        description={description}
                                                    />
                                                </Card>
                                            </Radio.Button>
                                        );
                                    })}
                                </Radio.Group>
                            </Form.Item>
                        </span>
                    </Form.Item>
                )
            }
        </Form.Item>
    );
};

export { FormContentBlockerFieldVisualType };
