import { Form, Input, Switch } from "antd";
import { useState } from "react";

import { useFormService } from "../../../../../contexts/formService.js";
import { useI18n } from "../../../../../contexts/i18n.js";
import { Notices } from "../../../../common/notices.js";
import { FormValueDifferFromTemplateTag } from "../../../valueDifferFromTemplateTag.js";

import type { FormServiceValueProps } from "../../../../../types/formService.js";
import type { FC } from "react";

const FormServiceFieldGeneralProvider: FC = () => {
    const { __, _i } = useI18n();
    const { template } = useFormService();
    const [ref, setRef] = useState<HTMLParagraphElement>();

    const isProviderCurrentWebsiteLabel = __(
        "I as the website operator am the provider/responsible for data processing (usually for self-hosted services)",
    );

    return (
        <Form.Item<FormServiceValueProps> label={__("Provider")} required>
            <Form.Item<FormServiceValueProps>
                noStyle
                shouldUpdate={(prevValues, nextValues) =>
                    prevValues.isProviderCurrentWebsite !== nextValues.isProviderCurrentWebsite
                }
            >
                {({ getFieldValue, setFieldValue }) => {
                    const isProviderCurrentWebsite = getFieldValue("isProviderCurrentWebsite") as boolean;

                    return (
                        <>
                            <div style={{ marginTop: 4 }}>
                                <Form.Item<FormServiceValueProps>
                                    name="isProviderCurrentWebsite"
                                    valuePropName="checked"
                                    noStyle
                                >
                                    <Switch />
                                </Form.Item>
                                <span>
                                    &nbsp;&nbsp;
                                    {isProviderCurrentWebsiteLabel}
                                </span>
                            </div>
                            <div
                                style={{ marginTop: 10, marginBottom: isProviderCurrentWebsite ? 0 : 10 }}
                                ref={setRef}
                            >
                                <FormValueDifferFromTemplateTag
                                    form="service"
                                    valueName="isProviderCurrentWebsite"
                                    widthOfRef={ref}
                                    renderDiff={(value) => (
                                        <>
                                            <Switch disabled checked={value} />
                                            <span>
                                                &nbsp;&nbsp;
                                                {isProviderCurrentWebsiteLabel}
                                            </span>
                                        </>
                                    )}
                                    noBr
                                />
                            </div>
                            <Form.Item<FormServiceValueProps>
                                name="provider"
                                noStyle
                                rules={[
                                    { required: !isProviderCurrentWebsite, message: __("Please name a provider!") },
                                ]}
                                dependencies={["isProviderCurrentWebsite"]}
                            >
                                <Input.TextArea
                                    autoSize
                                    style={{ display: isProviderCurrentWebsite ? "none" : undefined }}
                                    placeholder={__("e.g. Global Co., 90210 Broadway Blvd., Nashville, TN 37011-5678")}
                                    onKeyPress={(e) => {
                                        // Do not allow paragraphs
                                        if (e.keyCode === 13 || e.which === 13) {
                                            e.preventDefault();
                                            return false;
                                        }

                                        return undefined;
                                    }}
                                    onPaste={(e) => {
                                        // Remove paragraphs when pasted through clipboard
                                        setTimeout(() => {
                                            const { value } = e.target as HTMLTextAreaElement;
                                            setFieldValue("provider", value.replace(/\n+/gm, ", "));
                                        }, 0);
                                    }}
                                />
                            </Form.Item>
                            <p className="description" ref={setRef}>
                                {!isProviderCurrentWebsite && (
                                    <FormValueDifferFromTemplateTag
                                        form="service"
                                        valueName="provider"
                                        widthOfRef={ref}
                                        renderDiff={(value) => <Input value={value} readOnly />}
                                    />
                                )}
                                {_i(
                                    __(
                                        'Services have an "owner" who processes personal data or information saved in cookies etc. For self-hosted services, you are the data processor. Your contact details under {{a}}Settings > General > Website operator details{{/a}} are used. For other services, input the provider\'s contact details, often found in their privacy policy.',
                                    ),
                                    {
                                        a: <a href="#/settings" target="_blank" style={{ fontStyle: "italic" }} />,
                                    },
                                )}
                            </p>
                            <Notices
                                notices={[
                                    {
                                        message: template?.providerNotice,
                                        severity: "info",
                                    },
                                ]}
                            />
                        </>
                    );
                }}
            </Form.Item>
        </Form.Item>
    );
};

export { FormServiceFieldGeneralProvider };
