import { Checkbox, Form, Input, Select, Tooltip } from "antd";
import { useMemo } from "react";

import { useI18n } from "../../../../../../../contexts/i18n.js";
import { FormServiceFieldTechnicalDefinitionsLayout } from "../table.js";

import type { FormServiceValueProps } from "../../../../../../../types/formService.js";
import type { FormListFieldData } from "../table.js";
import type { FC } from "react";

const FormServiceFieldTechnicalDefinitionsCookieDuration: FC<{ field: FormListFieldData; readOnly?: boolean }> = ({
    field,
    readOnly,
}) => {
    const { __ } = useI18n();

    const durations = useMemo(
        () => ({
            s: __("second(s)"),
            m: __("minute(s)"),
            h: __("hour(s)"),
            d: __("day(s)"),
            mo: __("month(s)"),
            y: __("year(s)"),
        }),
        [__],
    );

    return (
        <Form.Item<FormServiceValueProps>
            noStyle
            shouldUpdate={(prevValues, nextValues) =>
                prevValues.technicalDefinitions[field.name]?.type !==
                    nextValues.technicalDefinitions[field.name]?.type ||
                prevValues.technicalDefinitions[field.name]?.isSessionDuration !==
                    nextValues.technicalDefinitions[field.name]?.isSessionDuration
            }
        >
            {({ getFieldValue }) => {
                const type = getFieldValue([
                    "technicalDefinitions",
                    field.name,
                    "type",
                ]) as FormServiceValueProps["technicalDefinitions"][0]["type"];
                const isSessionDuration = getFieldValue([
                    "technicalDefinitions",
                    field.name,
                    "isSessionDuration",
                ]) as FormServiceValueProps["technicalDefinitions"][0]["isSessionDuration"];

                if (["local", "session", "indexedDb", "flash"].indexOf(type) > -1) {
                    return null;
                }

                return (
                    <>
                        <Form.Item<FormServiceValueProps["technicalDefinitions"]>
                            name={[field.name, "isSessionDuration"]}
                            noStyle
                            valuePropName="checked"
                        >
                            <Checkbox style={{ float: "left", marginTop: 5 }} disabled={readOnly}>
                                <Tooltip title={__("This cookie is active as long as the session is active")}>
                                    <span>{__("Session")}</span>
                                </Tooltip>
                            </Checkbox>
                        </Form.Item>
                        <Form.Item<FormServiceValueProps["technicalDefinitions"]>
                            {...FormServiceFieldTechnicalDefinitionsLayout}
                            name={[field.name, "duration"]}
                            rules={[
                                {
                                    required: !isSessionDuration,
                                    message: __("Please provide a valid duration!"),
                                },
                            ]}
                            dependencies={[["technicalDefinitions", field.name, "isSessionDuration"]]}
                        >
                            <Input
                                disabled={readOnly}
                                min="0"
                                addonAfter={
                                    <Form.Item<FormServiceValueProps["technicalDefinitions"]>
                                        name={[field.name, "durationUnit"]}
                                        noStyle
                                        rules={[
                                            {
                                                required: !isSessionDuration,
                                                message: __("Please provide an unit!"),
                                            },
                                        ]}
                                    >
                                        <Select disabled={readOnly}>
                                            {Object.keys(durations).map((key) => (
                                                <Select.Option key={key} value={key}>
                                                    {durations[key]}
                                                </Select.Option>
                                            ))}
                                        </Select>
                                    </Form.Item>
                                }
                                type="number"
                                style={{
                                    maxWidth: 200,
                                    display: isSessionDuration ? "none" : undefined,
                                }}
                            />
                        </Form.Item>
                    </>
                );
            }}
        </Form.Item>
    );
};

export { FormServiceFieldTechnicalDefinitionsCookieDuration };
