import { Form, Input } from "antd";

import { isHost, isUrl } from "@devowl-wp/react-utils";

import { useI18n } from "../../../../../../../contexts/i18n.js";
import { Notices } from "../../../../../../common/notices.js";
import { FormServiceFieldTechnicalDefinitionsLayout } from "../table.js";

import type { FormServiceValueProps } from "../../../../../../../types/formService.js";
import type { FormListFieldData } from "../table.js";
import type { FC } from "react";

const FormServiceFieldTechnicalDefinitionsCookieHost: FC<{ field: FormListFieldData; readOnly?: boolean }> = ({
    field,
    readOnly,
}) => {
    const { __, _i } = useI18n();

    return (
        <>
            <Form.Item<FormServiceValueProps>
                noStyle
                shouldUpdate={(prevValues, nextValues) =>
                    prevValues.technicalDefinitions[field.name]?.type !==
                        nextValues.technicalDefinitions[field.name]?.type ||
                    prevValues.technicalDefinitions[field.name]?.host !==
                        nextValues.technicalDefinitions[field.name]?.host
                }
            >
                {({ getFieldValue }) => {
                    const type = getFieldValue([
                        "technicalDefinitions",
                        field.name,
                        "type",
                    ]) as FormServiceValueProps["technicalDefinitions"][0]["type"];
                    const host = getFieldValue([
                        "technicalDefinitions",
                        field.name,
                        "host",
                    ]) as FormServiceValueProps["technicalDefinitions"][0]["host"];

                    return (
                        <>
                            <Form.Item<FormServiceValueProps["technicalDefinitions"]>
                                {...FormServiceFieldTechnicalDefinitionsLayout}
                                name={[field.name, "host"]}
                                rules={[
                                    {
                                        validator: (_, value: string) =>
                                            (["local", "session", "indexedDb"].indexOf(type) > -1
                                                ? isUrl(value)
                                                : isHost(value)) ||
                                            (value === "" && type !== "http")
                                                ? Promise.resolve()
                                                : Promise.reject(__("Please provide a valid hostname!")),
                                    },
                                ]}
                                dependencies={[["technicalDefinitions", field.name, "type"]]}
                            >
                                <Input disabled={readOnly} />
                            </Form.Item>
                            {host.startsWith("*.") && host.length > 2 && (
                                <Notices
                                    notices={[
                                        {
                                            message: _i(
                                                __(
                                                    "You are using an invalid wildcard (placeholder) syntax {{code}}*.{{/code}} to match subdomains. Use {{code}}.%s{{/code}} to include subdomains.",
                                                    host.substr(2),
                                                ),
                                                {
                                                    code: <code />,
                                                },
                                            ),
                                            severity: "warning",
                                        },
                                    ]}
                                />
                            )}
                        </>
                    );
                }}
            </Form.Item>
        </>
    );
};

export { FormServiceFieldTechnicalDefinitionsCookieHost };
