import { Form, Popconfirm, Radio } from "antd";

import type { ServiceTagManager } from "@devowl-wp/cookie-consent-web-client";

import { useI18n } from "../../../contexts/i18n.js";
import { useUpsell } from "../../../contexts/upsell.js";
import { LearnMoreTag } from "../../common/learnMoreTag.js";
import { UpsellTag } from "../../upsell/tag.js";

import type { FormSettingsValueProps } from "../../../types/formSettings.js";
import type { FC } from "react";

const FormSettingsGoogleConsentModeTabContent: FC = () => {
    const { __, _i } = useI18n();
    const { isPro } = useUpsell();

    return (
        <>
            <Form.Item<FormSettingsValueProps> label={__("Google Consent Mode")} style={{ marginBottom: 10 }}>
                <Form.Item<FormSettingsValueProps> name="isGcm" noStyle>
                    <Radio.Group>
                        <Radio.Button disabled={!isPro} value={true}>
                            {__("Enabled")}
                        </Radio.Button>
                        <Radio.Button disabled={!isPro} value={false}>
                            {__("Disabled")}
                        </Radio.Button>
                    </Radio.Group>
                </Form.Item>
                <div style={{ margin: "5px 0px 0px" }}>
                    <LearnMoreTag
                        url={__("https://devowl.io/knowledge-base/real-cookie-banner-google-consent-mode-setup/")}
                    />
                    <UpsellTag
                        title={__("Enable Google Consent Mode?")}
                        testDrive
                        feature="gcm"
                        assetName={__("pro-modal/gcm.png")}
                        description={__(
                            "Google Consent Mode is a standard from Google for the standardized signaling of consent purposes. This is particularly important in order to be able to collect and evaluate data collected in Google services more comprehensively.",
                        )}
                    />
                </div>
                <p className="description">
                    {__(
                        "Google Consent Mode provides standardized consent for Google services to Google. This allows you e.g. to instruct Google not to set cookies for certain services, but still to collect basic data about your website visitors using other technologies. According to Google, only aggregated and anonymized data will be collected.",
                    )}
                </p>
            </Form.Item>
            <Form.Item<FormSettingsValueProps>
                noStyle
                shouldUpdate={(prevValues, curValues) =>
                    prevValues.isGcm !== curValues.isGcm ||
                    prevValues.setCookiesViaManager !== curValues.setCookiesViaManager
                }
            >
                {({ getFieldValue, setFieldsValue }) => {
                    const isGcm = getFieldValue("isGcm") as boolean;
                    const setCookiesViaManager = getFieldValue("setCookiesViaManager") as ServiceTagManager;

                    return (
                        <>
                            {!isGcm && new Date() <= new Date("2024-05-30") && (
                                <Form.Item<FormSettingsValueProps>
                                    wrapperCol={{ offset: 6 }}
                                    style={{ marginBottom: 10 }}
                                >
                                    <div
                                        className="notice notice-info inline below-h2 notice-alt"
                                        style={{ margin: 0 }}
                                    >
                                        <p>
                                            {_i(
                                                __(
                                                    "As of March 2024, Google Consent Mode will be required as Google's response to the {{aDma}}Digital Markets Act (DMA){{/aDma}} in order to continue collecting complete data in e.g. Google Analytics, Remarketing for Google Ads or Google AdSense.",
                                                ),
                                                {
                                                    aDma: (
                                                        <a
                                                            href={__(
                                                                "https://eur-lex.europa.eu/legal-content/EN/TXT/?uri=CELEX:32019R1150",
                                                            )}
                                                            target="_blank"
                                                            rel="noreferrer"
                                                        />
                                                    ),
                                                },
                                            )}
                                        </p>
                                    </div>
                                </Form.Item>
                            )}
                            {isPro && isGcm && (
                                <>
                                    <Form.Item<FormSettingsValueProps>
                                        label={__("Google Tag Manager integration")}
                                        required
                                    >
                                        <Radio.Group
                                            value={setCookiesViaManager === "googleTagManagerWithGcm"}
                                            onChange={(e) => {
                                                if (e.target.value && setCookiesViaManager === "googleTagManager") {
                                                    // Handled through `Popconfirm`
                                                } else {
                                                    setFieldsValue({
                                                        setCookiesViaManager: e.target.value
                                                            ? "googleTagManagerWithGcm"
                                                            : "none",
                                                    });
                                                }
                                            }}
                                        >
                                            {setCookiesViaManager === "googleTagManager" ? (
                                                <Popconfirm
                                                    title={
                                                        <>
                                                            <p style={{ marginTop: 0 }}>
                                                                {__(
                                                                    "By activating this Google Tag Manager integration, consents will be transferred to Google Tag Manager via Google Consent Mode from now on. Please note that you have previously transferred consents to Google Tag Manager via Google Tag Manager Events. The Google Tag Manager events will no longer be transmitted after changing this setting.",
                                                                )}
                                                            </p>
                                                            <p>
                                                                {_i(
                                                                    __(
                                                                        "You must therefore {{a}}reconfigure the settings in your Google Tag Manager for Google Consent Mode{{/a}}!",
                                                                    ),
                                                                    {
                                                                        a: (
                                                                            <a
                                                                                href={__(
                                                                                    "https://devowl.io/knowledge-base/real-cookie-banner-with-google-tag-manager-google-consent-mode/",
                                                                                )}
                                                                                target="_blank"
                                                                                rel="noreferrer"
                                                                            />
                                                                        ),
                                                                    },
                                                                )}
                                                            </p>
                                                        </>
                                                    }
                                                    cancelText={__("Cancel")}
                                                    okText={__("Activate now!")}
                                                    overlayStyle={{ maxWidth: 450 }}
                                                    onConfirm={() => {
                                                        setFieldsValue({
                                                            setCookiesViaManager: "googleTagManagerWithGcm",
                                                        });
                                                    }}
                                                    placement="bottomLeft"
                                                >
                                                    <Radio.Button value={true}>{__("Enabled")}</Radio.Button>
                                                </Popconfirm>
                                            ) : (
                                                <Radio.Button value={true}>{__("Enabled")}</Radio.Button>
                                            )}
                                            <Radio.Button value={false}>{__("Disabled")}</Radio.Button>
                                        </Radio.Group>
                                        <div style={{ margin: "5px 0px 0px" }}>
                                            <LearnMoreTag
                                                url={__(
                                                    "https://devowl.io/knowledge-base/real-cookie-banner-with-google-tag-manager-google-consent-mode/",
                                                )}
                                            />
                                        </div>
                                        <p className="description">
                                            {__(
                                                'Do you use Google Tag Manager? If yes, please activate this option to transfer consents for all services that you obtain in your cookie banner as "Additional consent" via Google Consent Mode to the consent management of Google Tag Manager.',
                                            )}
                                        </p>
                                    </Form.Item>
                                    <Form.Item<FormSettingsValueProps>
                                        label={__("Show recommendations for using Google services without consent")}
                                        required
                                    >
                                        <Form.Item<FormSettingsValueProps>
                                            noStyle
                                            name="isGcmShowRecommandationsWithoutConsent"
                                            rules={[
                                                {
                                                    type: "boolean",
                                                    required: true,
                                                    message: __("Please choose an option!"),
                                                },
                                            ]}
                                        >
                                            <Radio.Group>
                                                <Radio.Button value={true}>{__("Enabled")}</Radio.Button>
                                                <Radio.Button value={false}>{__("Disabled")}</Radio.Button>
                                            </Radio.Group>
                                        </Form.Item>
                                        <div style={{ margin: "5px 0px 0px" }}>
                                            <LearnMoreTag
                                                url={__(
                                                    "https://devowl.io/knowledge-base/real-cookie-banner-google-consent-mode-setup/#data-protection-criticism-of-google-consent-mode",
                                                )}
                                            />
                                        </div>
                                        <p className="description">
                                            {_i(
                                                __(
                                                    "At its core, Google Consent Mode wants to collect more data about website visitors for you, for which Google services should also be loaded before consent has been given. In this case, Google for example prevents the setting of cookies. However, the IP address of your website operator is already transmitted to Google. If the IP address is regarded as personal data (usually common legal interpretation), this requires a legal basis in accordance with {{aGdpr}}Art. 6 GDPR{{/aGdpr}}. Apart from consent, only legitimate interest comes into question, which must be viewed critically in this case in terms of data protection law. First and foremost, you as the website operator and not Google are liable for possible data protection violations of this kind. By activating this option, Real Cookie Banner will show you as website operator, recommendations for the configuration of services which, in our legal opinion, lead to data protection violations, but which are in the interest of using Google Consent Mode. Google Consent Mode can also be used if the services are only loaded with consent.",
                                                ),
                                                {
                                                    aGdpr: (
                                                        <a
                                                            href={__("https://gdpr-text.com/read/article-6/")}
                                                            target="_blank"
                                                            rel="noreferrer"
                                                        />
                                                    ),
                                                },
                                            )}
                                        </p>
                                    </Form.Item>
                                    <Form.Item<FormSettingsValueProps>
                                        label={__("Collect additional data via URL parameters")}
                                    >
                                        <Form.Item<FormSettingsValueProps>
                                            noStyle
                                            name="isGcmCollectAdditionalDataViaUrlParameters"
                                        >
                                            <Radio.Group>
                                                <Radio.Button value={true}>{__("Enabled")}</Radio.Button>
                                                <Radio.Button value={false}>{__("Disabled")}</Radio.Button>
                                            </Radio.Group>
                                        </Form.Item>
                                        <div style={{ margin: "5px 0px 0px" }}>
                                            <LearnMoreTag
                                                url={__(
                                                    "https://devowl.io/knowledge-base/real-cookie-banner-google-consent-mode-setup/#collect-additional-data-via-url-parameters",
                                                )}
                                            />
                                        </div>
                                        <p className="description">
                                            {_i(
                                                __(
                                                    "If consent is not given for the {{code}}ad_storage{{/code}} and/or {{code}}analytics_storage{{/code}} consnt types (purposes) of Google Consent Mode, this option can be used to transfer data that is typically stored in cookies as GET parameters in internal links. This is particularly relevant if you place advertisements in Google Ads and conversion tracking should also work without consent. Please note that data can only be collected if you download the respective Google services before obtaining consent. As client IDs can be transmitted, among other things, which can be regarded as personal data, we consider this data transmission to be questionable in terms of data protection law.",
                                                ),
                                                {
                                                    code: <code />,
                                                },
                                            )}
                                        </p>
                                    </Form.Item>
                                    <Form.Item<FormSettingsValueProps> label={__("Redact ads data without consent")}>
                                        <Form.Item<FormSettingsValueProps>
                                            noStyle
                                            name="isGcmRedactAdsDataWithoutConsent"
                                        >
                                            <Radio.Group>
                                                <Radio.Button value={true}>{__("Enabled")}</Radio.Button>
                                                <Radio.Button value={false}>{__("Disabled")}</Radio.Button>
                                            </Radio.Group>
                                        </Form.Item>
                                        <div style={{ margin: "5px 0px 0px" }}>
                                            <LearnMoreTag
                                                url={__(
                                                    "https://devowl.io/knowledge-base/real-cookie-banner-google-consent-mode-setup/#redact-ads-data-without-consent",
                                                )}
                                            />
                                        </div>
                                        <p className="description">
                                            {_i(
                                                __(
                                                    "If consent is not given for the {{code}}ad_storage{{/code}} consent type (purpose) of Google Consent Mode, no new cookies will be set for advertising purposes, except to fight fraud and spam. In addition, this option allows measurements for clicks on advertising to be routed through the domain {{code}}googleadsyndication.com{{/code}} instead of {{code}}doubleclick.net{{/code}} or {{code}}google.com{{/code}}, on which no cookies are set for advertising, and identifiers for clicks and page URLs are removed from advertising links. This enables more data protection-friendly conversion tracking with the simultaneous loss of detailed information on the attribution of target achievement.",
                                                ),
                                                {
                                                    code: <code />,
                                                },
                                            )}
                                        </p>
                                    </Form.Item>
                                    <Form.Item<FormSettingsValueProps>
                                        label={__("Naming of requested consent types in first view")}
                                    >
                                        <Form.Item<FormSettingsValueProps> noStyle name="isGcmListPurposes">
                                            <Radio.Group>
                                                <Radio.Button value={true}>{__("Enabled")}</Radio.Button>
                                                <Radio.Button value={false}>{__("Disabled")}</Radio.Button>
                                            </Radio.Group>
                                        </Form.Item>
                                        <div style={{ margin: "5px 0px 0px" }}>
                                            <LearnMoreTag
                                                url={__(
                                                    "https://devowl.io/knowledge-base/real-cookie-banner-google-consent-mode-setup/#naming-of-requested-consent-types-in-first-view",
                                                )}
                                            />
                                        </div>
                                        <p className="description">
                                            {__(
                                                "The visitor to your website should be adequately informed as to which consent types (purposes) they are consenting to in accordance with Google Consent Mode. For this reason, all consent types should be listed by name in the first view of the cookie banner. This makes it easier for your website visitor to assess whether they really want to agree to all consents without visiting the individual privacy settings. ",
                                            )}
                                        </p>
                                    </Form.Item>
                                </>
                            )}
                        </>
                    );
                }}
            </Form.Item>
        </>
    );
};

export { FormSettingsGoogleConsentModeTabContent };
