import { WarningOutlined } from "@ant-design/icons";
import { Checkbox, Divider, Form, Popconfirm, Space, Tooltip } from "antd";
import { useMemo, useState } from "react";

import { useFormSettings } from "../../../contexts/formSettings.js";
import { useI18n } from "../../../contexts/i18n.js";

import type { FC } from "react";

const FormSettingsResetTabContent: FC = () => {
    const { __ } = useI18n();
    const { onResetAll, onResetTexts, resetTextsLanguages } = useFormSettings();
    const [resetConsents, setResetConsents] = useState(false);
    const defaultChecked = useMemo(
        () => resetTextsLanguages.filter((item) => item.isRequired).map((item) => item.code),
        [resetTextsLanguages],
    );
    const [resetTextsLanguagesChecked, setResetTextsLanguagesChecked] = useState<string[]>(defaultChecked);

    return (
        <>
            <Form.Item label={__("Reset texts")}>
                <p className="description" style={{ marginTop: 6 }}>
                    {__(
                        "You can reset all texts in your cookie banner to their default values. This includes texts in the customizer (customize banner), service groups, services (if created from the service template), content blockers (if created from the content blocker template) and the cookie policy (if created).",
                    )}
                    {resetTextsLanguages.length > 1 && (
                        <>
                            &nbsp;
                            {__("You can select multiple languages for which you want to reset the texts.")}
                        </>
                    )}
                </p>
                {resetTextsLanguages.length !== 1 && (
                    <Space direction="vertical">
                        <Space direction="horizontal">
                            <a
                                href="#"
                                onClick={(e) => {
                                    e.preventDefault();
                                    setResetTextsLanguagesChecked(resetTextsLanguages.map((item) => item.code));
                                }}
                            >
                                {__("Select all")}
                            </a>
                            <Divider type="vertical" />
                            <a
                                href="#"
                                onClick={(e) => {
                                    e.preventDefault();
                                    setResetTextsLanguagesChecked(defaultChecked);
                                }}
                            >
                                {__("Deselect all")}
                            </a>
                        </Space>
                        <Checkbox.Group
                            options={resetTextsLanguages.map(({ code, name, isDisabled, isRequired, notice }) => ({
                                label: notice ? (
                                    <Tooltip title={notice}>
                                        {name}&nbsp;
                                        <WarningOutlined />
                                    </Tooltip>
                                ) : (
                                    name
                                ),
                                value: code,
                                disabled: isDisabled || isRequired,
                            }))}
                            value={resetTextsLanguagesChecked}
                            onChange={(value) => setResetTextsLanguagesChecked(value)}
                        />
                    </Space>
                )}
                <div style={{ marginTop: 10 }}>
                    <Popconfirm
                        title={__("Are you sure?")}
                        cancelText={__("Cancel")}
                        okText={__("Reset texts")}
                        onConfirm={() => onResetTexts(resetTextsLanguagesChecked)}
                        placement="topLeft"
                    >
                        <input type="button" className="button" value={__("Reset")} />
                    </Popconfirm>
                </div>
            </Form.Item>
            <Form.Item label={__("Reset all")}>
                <p className="description" style={{ marginTop: 6 }}>
                    {__(
                        "You can reset all settings of Real Cookie Banner to the default values. This will also irreversibly delete any services and content blockers that have been created, as well as design settings. Documentation of consent may be kept to fulfill the legal disclosure requirement.",
                    )}
                </p>
                <Checkbox checked={resetConsents} onChange={(e) => setResetConsents(e.target.checked)}>
                    {__("Delete documented consent")}
                </Checkbox>
                <div style={{ marginTop: 10 }}>
                    <Popconfirm
                        title={__("Are you sure?")}
                        cancelText={__("Cancel")}
                        okText={__("Reset all")}
                        onOpenChange={(open) => !open && setResetConsents(false)}
                        onConfirm={() => onResetAll(resetConsents)}
                        placement="topLeft"
                    >
                        <input type="button" className="button" value={__("Reset")} />
                    </Popconfirm>
                </div>
            </Form.Item>
        </>
    );
};

export { FormSettingsResetTabContent };
