import { InfoCircleOutlined } from "@ant-design/icons";
import { Empty, Table, Tag, Tooltip } from "antd";
import { useMemo } from "react";

import { FORM_TCF_VENDOR_FIELD_PURPOSES_COLUMN_NAME_WIDTH } from "./purposes.js";
import { useFormTcfVendor } from "../../../../contexts/formTcfVendor.js";
import { useI18n } from "../../../../contexts/i18n.js";
import { FormHeadline } from "../../headline.js";
import { FormLayoutTcfVendor } from "../../layout.js";

import type { FC } from "react";

const { Column } = Table;

const FormTcfVendorFieldFeatures: FC = () => {
    const { __ } = useI18n();

    const {
        vendor,
        declarations: { features, specialFeatures },
    } = useFormTcfVendor();

    const useFeatures = useMemo(
        () =>
            Object.values(features)
                .map((feature) =>
                    vendor.features.indexOf(feature.id) > -1 ? { feature, isSpecial: false } : undefined,
                )
                .concat(
                    Object.values(specialFeatures).map((specialFeature) =>
                        vendor.specialFeatures.indexOf(specialFeature.id) > -1
                            ? { feature: specialFeature, isSpecial: true }
                            : undefined,
                    ),
                )
                .filter(Boolean),
        [vendor, features, specialFeatures],
    );

    type TableRowType = (typeof useFeatures)[0];

    return (
        <>
            <FormHeadline
                offset={FormLayoutTcfVendor.labelCol.span}
                description={__(
                    "Features are specified by the vendor and are immutable. They describe the characteristics of how personal data is processed in order to achieve one or more purposes.",
                )}
            >
                {__("Features and special features")}
            </FormHeadline>
            <Table<TableRowType>
                locale={{ emptyText: <Empty description={__("This vendor has not listed any used features.")} /> }}
                dataSource={useFeatures}
                rowKey={({ feature: { id }, isSpecial }) => `${id}-${isSpecial}`}
                size="small"
                bordered
                pagination={false}
                style={{ marginBottom: 25 }}
            >
                <Column<TableRowType>
                    width={FORM_TCF_VENDOR_FIELD_PURPOSES_COLUMN_NAME_WIDTH}
                    title={__("Name")}
                    dataIndex="name"
                    key="name"
                    render={(_, { feature: { name } }) => name}
                />
                <Column<TableRowType>
                    title={__("Description")}
                    dataIndex="description"
                    key="description"
                    render={(_, { feature: { description, illustrations }, isSpecial }) => (
                        <Tooltip
                            title={
                                illustrations.length > 0 && (
                                    <div>
                                        <strong>{__("Illustrations")}: </strong>
                                        <ul>
                                            {illustrations.map((illustration) => (
                                                <li key={illustration}>{illustration}</li>
                                            ))}
                                        </ul>
                                    </div>
                                )
                            }
                            placement="bottomLeft"
                            overlayInnerStyle={{ width: 500 }}
                        >
                            <InfoCircleOutlined style={{ cursor: "pointer", opacity: 0.5 }} />
                            &nbsp;
                            {description} {isSpecial && <Tag color="warning">{__("Special feature")}</Tag>}
                        </Tooltip>
                    )}
                />
            </Table>
        </>
    );
};

export { FormTcfVendorFieldFeatures };
