import { useCallback } from "react";

import type { CommonContextProps } from "@devowl-wp/react-cookie-banner";

import { useI18n } from "../contexts/i18n.js";

import type { FormSettingsValueProps } from "../types/formSettings.js";

function useTerritorialLegalBasisArticles({
    predefinedDataProcessingInSafeCountriesLists,
    iso3166OneAlpha2,
}: {
    predefinedDataProcessingInSafeCountriesLists: CommonContextProps["predefinedDataProcessingInSafeCountriesLists"];
    iso3166OneAlpha2: Record<string, string>;
}) {
    const { _x, __ } = useI18n();

    const getLabelForPredefinedCountryList = useCallback(
        (code: keyof typeof predefinedDataProcessingInSafeCountriesLists) => {
            const countries = predefinedDataProcessingInSafeCountriesLists[code]
                .map((cc) => iso3166OneAlpha2[cc])
                .join(", ");

            if (code === "ADEQUACY_EU") {
                return __("Countries with an adequacy decision in EU: %s", countries);
            } else if (code === "ADEQUACY_CH") {
                return __("Countries with an adequacy decision in Switzerland: %s", countries);
            } else if (code === "GDPR") {
                return __("Countries where GDPR applies: %s", countries);
            }
            return __("Countries where %s applies", code);
        },
        [__, predefinedDataProcessingInSafeCountriesLists, iso3166OneAlpha2],
    );

    return useCallback(
        (territorialLegalBasis: FormSettingsValueProps["territorialLegalBasis"]) => {
            const legalBasis: string[] = [];
            const dataProcessingInUnsafeCountriesArticles: string[] = [];
            const dataProcessingInUnsafeCountriesArticlesLinks: string[] = [];
            const safeCountriesList: Partial<
                Record<keyof typeof predefinedDataProcessingInSafeCountriesLists | "SAFE", string>
            > = {
                ADEQUACY_CH: getLabelForPredefinedCountryList("ADEQUACY_CH"),
                ADEQUACY_EU: getLabelForPredefinedCountryList("ADEQUACY_EU"),
                GDPR: getLabelForPredefinedCountryList("GDPR"),
                SAFE: "",
            };
            const adequacyCountriesLists: string[] = [];

            const isGdpr = territorialLegalBasis.indexOf("gdpr-eprivacy") > -1;
            const isDsg = territorialLegalBasis.indexOf("dsg-switzerland") > -1;

            const labelGdprEuEea = _x("GDPR (EU/EEA)", "legal-text");
            const labelDsg = _x("DSG (Switzerland)", "legal-text");
            const labelGdpr = _x("GDPR", "legal-text");

            const dataProcessInUnsafeCountriesArticleGdprEuEea = _x("Art. 49 (1) (a) GDPR (EU/EEA)", "legal-text");
            const dataProcessInUnsafeCountriesArticleDsg = _x("Art. 17 (1) (a) DSG (Switzerland)", "legal-text");
            const dataProcessInUnsafeCountriesArticleGdpr = _x("Art. 49 (1) (a) GDPR", "legal-text");

            const dataProcessInUnsafeCountriesArticleLinkGdpr = __("https://gdpr-text.com/read/article-49/");
            const dataProcessInUnsafeCountriesArticleLinkDsg = __(
                "https://www.fedlex.admin.ch/eli/cc/2022/491/en#art_17",
            );

            if (isDsg && isGdpr) {
                legalBasis.push(labelGdprEuEea, labelDsg);
                dataProcessingInUnsafeCountriesArticles.push(
                    dataProcessInUnsafeCountriesArticleGdprEuEea,
                    dataProcessInUnsafeCountriesArticleDsg,
                );
                dataProcessingInUnsafeCountriesArticlesLinks.push(
                    dataProcessInUnsafeCountriesArticleLinkGdpr,
                    dataProcessInUnsafeCountriesArticleLinkDsg,
                );
                adequacyCountriesLists.push(safeCountriesList.ADEQUACY_EU, safeCountriesList.ADEQUACY_CH);

                const intersectionGdprAndAdequacyEuAndCh = predefinedDataProcessingInSafeCountriesLists.GDPR.filter(
                    (cc) => predefinedDataProcessingInSafeCountriesLists.ADEQUACY_CH.includes(cc),
                );
                if (intersectionGdprAndAdequacyEuAndCh.indexOf("CH") === -1) {
                    intersectionGdprAndAdequacyEuAndCh.push("CH");
                }
                safeCountriesList.SAFE = intersectionGdprAndAdequacyEuAndCh
                    .map((cc) => iso3166OneAlpha2[cc])
                    .join(", ");
            } else if (isDsg) {
                legalBasis.push(labelDsg);
                dataProcessingInUnsafeCountriesArticles.push(dataProcessInUnsafeCountriesArticleDsg);
                dataProcessingInUnsafeCountriesArticlesLinks.push(dataProcessInUnsafeCountriesArticleLinkDsg);
                adequacyCountriesLists.push(safeCountriesList.ADEQUACY_CH);
                safeCountriesList.SAFE = safeCountriesList.ADEQUACY_CH;
            } else if (isGdpr) {
                legalBasis.push(labelGdpr);
                dataProcessingInUnsafeCountriesArticles.push(dataProcessInUnsafeCountriesArticleGdpr);
                dataProcessingInUnsafeCountriesArticlesLinks.push(dataProcessInUnsafeCountriesArticleLinkGdpr);
                adequacyCountriesLists.push(safeCountriesList.ADEQUACY_EU);
                safeCountriesList.SAFE = `${safeCountriesList.GDPR}\n\n${safeCountriesList.ADEQUACY_EU}`;
            }

            return {
                legalBasis,
                dataProcessingInUnsafeCountriesArticles,
                dataProcessingInUnsafeCountriesArticlesLinks,
                isGdpr,
                isDsg,
                labelGdprEuEea,
                labelDsg,
                labelGdpr,
                safeCountriesList,
                adequacyCountriesLists,
            };
        },
        [_x, __, getLabelForPredefinedCountryList, getLabelForPredefinedCountryList],
    );
}

export { useTerritorialLegalBasisArticles };
