import { Fragment, useRef, useState } from "react";

import { extendBlockerHeroStylesheet, extendCommonButtonsStylesheet } from "@devowl-wp/web-cookie-banner";

import { BlockerHeroAudioWaveFromText } from "./hero/audio/waveFromText.js";
import { BlockerHeroDialog } from "./hero/dialog.js";
import { useBlocker } from "../../contexts/blocker.js";
import { Button } from "../common/button.js";
import { BlockerHeroVideoSvgPlayButton } from "./hero/video/svgPlayButton.js";
import { useBlockerStylesheet } from "../../hooks/blocker/useBlockerStylesheet.js";

import type { FC, HTMLAttributes } from "react";

const BlockerHeroContainer: FC<
    { blockerClassName: string; blockerOverlayId: string } & HTMLAttributes<HTMLDivElement>
> = ({ blockerClassName, blockerOverlayId, ...rest }) => {
    const stylesheet = useBlockerStylesheet();
    const {
        HeroWrapper,
        HeroBackgroundWrapper,
        HeroBackground,
        HeroTitle,
        HeroContainer,
        heroButton,
        HeroPlayButton,
        screenReaderOnlyClass,
    } = stylesheet.extend(...extendBlockerHeroStylesheet).extend(...extendCommonButtonsStylesheet);
    const blocker = useBlocker();
    const {
        texts: { blockerHeadline },
        blocker: { presetId, name, visualHeroButtonText, visualContentType, visualThumbnail },
    } = blocker;
    const ref = useRef<HTMLDivElement>();
    const heroContainerRef = useRef<HTMLDivElement>();
    const [isHover, setIsHover] = useState(false);
    const { url } = visualThumbnail;
    const title = visualThumbnail.title || name;
    const blockerHeadlineWithName = blockerHeadline.replace(/{{name}}/gi, name);
    const audioWaveText = `${visualThumbnail.title || blockerHeadlineWithName}ThiS iS jUst ANother TEXT TO reACh minIMum length`;
    const showPlayButton = ["video-player", "audio-player"].indexOf(visualContentType) > -1;

    return (
        <Fragment>
            <HeroWrapper ref={ref} {...rest}>
                <HeroBackgroundWrapper aria-hidden>
                    <HeroBackground src={url} alt={title} />
                </HeroBackgroundWrapper>
                <HeroContainer
                    role="presentation"
                    onMouseEnter={() => setIsHover(true)}
                    onMouseLeave={() => setIsHover(false)}
                    ref={heroContainerRef}
                >
                    <HeroTitle aria-hidden>{title}</HeroTitle>
                    <a href="#" className={screenReaderOnlyClass} aria-label={`${blockerHeadlineWithName}, ${title}`}>
                        {title}
                    </a>
                    {visualContentType === "audio-player" && <BlockerHeroAudioWaveFromText text={audioWaveText} />}
                    {visualHeroButtonText && ["audio-player", "video-player"].indexOf(visualContentType) === -1 ? (
                        <Button type="acceptAll" forceShow busyOnClick={false} className={heroButton}>
                            &nbsp;&nbsp;{visualHeroButtonText}&nbsp;&nbsp;
                        </Button>
                    ) : (
                        showPlayButton &&
                        (presetId === "youtube" && visualContentType === "video-player" ? (
                            <BlockerHeroVideoSvgPlayButton fill={isHover ? "#ff0808" : `rgb(0 0 0 / 70%)`} />
                        ) : (
                            <HeroPlayButton />
                        ))
                    )}
                </HeroContainer>
            </HeroWrapper>
            <BlockerHeroDialog
                blockerClassName={blockerClassName}
                blockerOverlayId={blockerOverlayId}
                heroContainerRef={heroContainerRef}
            />
        </Fragment>
    );
};

export { BlockerHeroContainer };
