import { useMemo } from "react";

import type { ITcfGvlAdditionalInformation } from "@devowl-wp/api-real-cookie-banner";
import { EServiceTemplateDataProcessingInCountriesSpecialTreatment } from "@devowl-wp/api-real-cookie-banner";

import { calculateUnsafeCountries } from "../../others/calculateSafetyMechanismsFromSpecialTreatments.js";
import { useBannerOrBlocker } from "../useBannerOrBlocker.js";

/**
 * Calculate the labels for the "Appropriate safeguards" section in the cookie banner.
 *
 * @deprecated Only for backwards-compatibility rendering (designVersion < 10)
 */
function useAppropriateSafeguards({
    dataProcessingInCountries,
    specialTreatments,
    tcf = { internationalTransfers: false, transferMechanisms: [] },
}: {
    dataProcessingInCountries: string[];
    specialTreatments: EServiceTemplateDataProcessingInCountriesSpecialTreatment[];
    tcf?: Pick<ITcfGvlAdditionalInformation, "internationalTransfers" | "transferMechanisms">;
}) {
    const {
        designVersion,
        i18n: { safetyMechanisms, other },
        isDataProcessingInUnsafeCountries,
        dataProcessingInUnsafeCountriesSafeCountries,
        iso3166OneAlpha2,
    } = useBannerOrBlocker();
    const { internationalTransfers, transferMechanisms } = tcf;

    const dataProcessingInUnsafeCountries = useMemo(() => {
        const mapCountryIso = (c: string) => iso3166OneAlpha2[c] || c;
        return isDataProcessingInUnsafeCountries
            ? calculateUnsafeCountries({
                  dataProcessingInCountries,
                  safeCountries: dataProcessingInUnsafeCountriesSafeCountries,
                  specialTreatments,
                  isDisplay: true,
              }).map(mapCountryIso)
            : [];
    }, [
        isDataProcessingInUnsafeCountries,
        dataProcessingInUnsafeCountriesSafeCountries,
        specialTreatments,
        dataProcessingInCountries,
        iso3166OneAlpha2,
    ]) as string[];

    const appropriateSafeguards = useMemo(
        () => [
            ...new Set( // Remove duplicates
                [
                    specialTreatments.indexOf(
                        EServiceTemplateDataProcessingInCountriesSpecialTreatment.StandardContractualClauses,
                    ) > -1 && safetyMechanisms.standardContractualClauses,
                    // TCF-defined transfer mechanisms
                    designVersion > 6 &&
                        internationalTransfers &&
                        transferMechanisms.map((mechanism) => {
                            switch (mechanism) {
                                case "SCCs":
                                    return safetyMechanisms.standardContractualClauses;
                                case "Adequacy decision":
                                    return safetyMechanisms.adequacyDecision;
                                case "BCRs":
                                    return safetyMechanisms.bindingCorporateRules;
                                case "Other":
                                    return other;
                                default:
                                    return "";
                            }
                        }),
                ]
                    .flat()
                    .filter(Boolean),
            ),
        ],
        [specialTreatments, internationalTransfers, transferMechanisms],
    );

    return {
        dataProcessingInUnsafeCountries,
        appropriateSafeguards,
    };
}

export { useAppropriateSafeguards };
