import { Grid } from "gridjs";

import { waitObject } from "@devowl-wp/react-utils";
import { createPureScopedStylesheet, createStylesheet } from "@devowl-wp/web-scoped-css";

/**
 * Use the power of [Grid.js](https://www.npmjs.com/package/gridjs) to create a styled table for the cookie policy.
 *
 * It is highly recommend to import this module with a dynamic import in an own chunk.
 */
function createCookiePolicyTable(element: HTMLTableElement) {
    const isDarkMode = element.hasAttribute("data-gridjs-dark-mode");

    /**
     * > As a macOS user, I didn't understand that I need to scroll vertically to see the full table (it literally took me 5 minutes to understand this; lol).
     * > Is it possible to display, for example, a white overlay on the left/right side when I scroll (like at the bottom of the cookie banner)?
     * > **Or a permanently visible vertical scrollbar?**
     */
    const customCss = createPureScopedStylesheet(({ rule }) => {
        const [, container] = rule({
            pseudos: {
                " ::-webkit-scrollbar": {
                    appearance: "none",
                    width: "7px",
                    height: "7px",
                    background: isDarkMode ? "#2c2c2c" : undefined,
                },
                " ::-webkit-scrollbar-thumb": {
                    borderRadius: "4px",
                    background: isDarkMode ? "#3f3f3f" : "rgba(0,0,0,.5)",
                    boxShadow: "0 0 1px rgba(255,255,255,.5)",
                },
            },
        });

        const [, disableSortingDueToBug] = rule({
            pointerEvents: "none",
            pseudos: {
                ">.gridjs-sort": {
                    display: "none",
                },
            },
        });

        return { container, disableSortingDueToBug };
    });
    customCss.toggle(true);

    import(
        /* webpackChunkName: "cookie-policy", webpackMode: "lazy-once" */ "gridjs/dist/theme/mermaid.min.css?inline-raw"
    ).then(({ default: gridJs }) => createStylesheet(gridJs, { id: "grid-js", overwrite: false }));

    import(
        // eslint-disable-next-line path-import-extension/require-path-import-extension
        /* webpackChunkName: "cookie-policy", webpackMode: "lazy-once" */ "../style/gridjs-dark.min.css?inline-raw"
    ).then(({ default: gridJs }) => {
        if (isDarkMode) {
            createStylesheet(gridJs, { id: "grid-js-dark", overwrite: false });
        }
    });

    const language = JSON.parse((element.previousSibling as HTMLScriptElement).innerHTML);
    const {
        pagination: { navigate: navigateI18n, page: pageI18n },
    } = language;
    language.pagination.navigate = (page: number, pages: number) =>
        (navigateI18n as string).replace("%1$d", `${page}`).replace("%2$d", `${pages}`);
    language.pagination.page = (page: number) => (pageI18n as string).replace("%d", `${page}`);

    const wrapper = document.createElement("div");
    element.parentElement.insertBefore(wrapper, element);
    const grid = new Grid({
        className: {
            container: customCss.container,
        },
        from: element,
        sort: true,
        search: true,
        pagination: true,
        autoWidth: false,
        language,
    });
    grid.render(wrapper);

    // Disable sorting of "Duration" until https://github.com/grid-js/gridjs/issues/1461#issuecomment-2217587700 is solved
    waitObject(() => wrapper.querySelector(".gridjs-thead>tr>th:nth-child(5)"), 100, 20).then((e) =>
        e.classList.add(customCss.disableSortingDueToBug),
    );
}

export { createCookiePolicyTable };
