import { Component } from "react";

import { Icon, Spin, ToolbarButton } from "./index.js";

import type { ToolbarButtonProps } from "./ToolbarButton.js";
import type { TreeNodeProps } from "./TreeNode.js";

export interface ToolbarProps {
    isToolbarActive?: boolean;
    isToolbarBusy?: boolean;
    buttons?: { [key: string]: ToolbarButtonProps };
    backButton?: { label: any; save?: any };
    /**
     * The active toolbar button (must match the same as the object key of the button)
     */
    activeButton?: string;
    checkedNodes?: TreeNodeProps[];
    onCloseMultiToolbar?: (props: ToolbarButtonProps) => void;
}

/**
 * Renders the toolbar with button. If <code>props.activeButton</code> is
 * set all other <code>ToolbarButton</code>`s are disabled expect the <code>props.backButton</code> -
 * then the user clicks on the cancel button (<code>props.backButton.label</code>)
 * and the <code>onCancel</code> callback of the ToolbarButton gets executed.
 *
 * <p>If the ToolbarButton has a valid <code>onSave</code> callback the toolbar has an additional save button (<code>props.backButton.save</code>)
 * - then the user clicks on the save button and the callback gets executed.</p>
 *
 * @returns
 */
export class Toolbar extends Component<ToolbarProps> {
    public handleCancelClick = (cancelNode: ToolbarButtonProps) => {
        const originalNode = this.props.buttons[this.props.activeButton];
        originalNode && originalNode.onCancel && originalNode.onCancel(cancelNode, originalNode);
    };

    public render() {
        const { isToolbarActive, isToolbarBusy, buttons, backButton, activeButton, checkedNodes, onCloseMultiToolbar } =
            this.props;
        const activeActions = [];
        const hasActionButtonSave = activeButton && buttons[activeButton].onSave;

        activeButton &&
            activeActions.push(
                <ToolbarButton key="activeButtonCancel" content={backButton.label} onClick={this.handleCancelClick} />,
            );
        hasActionButtonSave &&
            activeActions.push(
                <ToolbarButton key="activeButtonSave" content={backButton.save} onClick={hasActionButtonSave} />,
            );

        return (
            // @ts-expect-error outdated @types/react
            <Spin spinning={isToolbarBusy} size="small">
                <div className="aiot-toolbar">
                    <div className="aiot-toolbar-items">
                        {activeButton
                            ? activeActions
                            : Object.keys(buttons).map((key) => {
                                  return (
                                      <ToolbarButton
                                          key={key}
                                          {...buttons[key]}
                                          disabled={!isToolbarActive ? true : buttons[key].disabled}
                                      />
                                  );
                              })}
                        {checkedNodes.length > 0 && [
                            <ToolbarButton
                                key="multiple-count"
                                cssClasses="aiot-border-left"
                                disabled
                                content={
                                    <span>
                                        <Icon type="check" /> {checkedNodes.length}
                                    </span>
                                }
                            />,
                            <ToolbarButton
                                key="multiple-close"
                                content={<Icon type="close" />}
                                onClick={onCloseMultiToolbar}
                            />,
                        ]}
                    </div>
                    <div className="clear"></div>
                </div>
            </Spin>
        );
    }
}
