import classNames from "classnames";
import { Component } from "react";

import { Dropdown, Menu, Tooltip } from "./index.js";

import type { TooltipProps } from "./Tooltip.js";
import type { TooltipPlacement } from "antd-legacy/lib/tooltip";
import type { MouseEvent } from "react";

export interface ToolbarButtonProps {
    visible?: boolean;
    cssClasses?: string;
    toolTipTitle?: TooltipProps["title"];
    /**
     * The tooltip text. If you want to show a tooltip you need to set a toolTipTitle, too.
     */
    toolTipText?: TooltipProps["content"];
    disabled?: boolean;
    /**
     * A menu implementation (see https://ant.design/components/menu/). You can wrap an menu item with Tooltip (do not forget css class <code>overlayClassName="aiot-toolbar-menu"</code>)
     */
    menu?: any;
    toolTipPlacement?: TooltipPlacement;
    dropdownPlacement?: "topLeft" | "topCenter" | "topRight" | "bottomLeft" | "bottomCenter" | "bottomRight";
    content?: any;
    onClick?: (props: ToolbarButtonProps) => void;
    onSave?: (props: ToolbarButtonProps) => void;
    onCancel?: (cancelNode: ToolbarButtonProps, originalNode: ToolbarButtonProps) => void;
    modifier?: (body: JSX.Element, props: ToolbarButtonProps) => JSX.Element;
}

/**
 * Toolbar button for the toolbar component.
 *
 * @returns
 */
export class ToolbarButton extends Component<ToolbarButtonProps> {
    public handleClick = (e: MouseEvent): void => {
        if (!this.props.disabled) {
            this.props.onClick && this.props.onClick(this.props);
        }
        e.preventDefault();
    };

    public render() {
        // @TODO <Menu onClick={handleMenuClick}
        const {
            visible = true,
            content,
            cssClasses,
            toolTipTitle,
            toolTipText,
            disabled,
            menu,
            toolTipPlacement,
            dropdownPlacement,
            modifier,
        } = this.props;
        const className = classNames(cssClasses, "aiot-tooltip", {
            "aiot-disabled": disabled,
        });
        const bodyAttr = { href: "#", className, disabled, onClick: this.handleClick };
        if (!visible) {
            return null;
        }
        let body = <a {...bodyAttr}>{content}</a>;

        modifier && (body = modifier(body, this.props));

        // Make tooltip
        if (toolTipTitle && toolTipText) {
            body = (
                <Tooltip
                    placement={toolTipPlacement ? toolTipPlacement : menu ? "rightTop" : "bottom"}
                    title={toolTipTitle}
                    content={toolTipText}
                >
                    {body}
                </Tooltip>
            );
        }

        // Make dropdown with menu items
        if (menu) {
            let overlay = menu;
            if (menu.map) {
                const overlayMenu = menu.map(({ key, label, toolTipTitle, toolTipText, ...rest }: any) => {
                    const menuItem = (
                        <Menu.Item key={key} {...rest}>
                            {label}
                        </Menu.Item>
                    );
                    return toolTipTitle && toolTipText ? (
                        <Tooltip
                            overlayClassName="aiot-toolbar-menu"
                            placement="rightTop"
                            title={toolTipTitle}
                            content={toolTipText}
                        >
                            {menuItem}
                        </Tooltip>
                    ) : (
                        menuItem
                    );
                });
                overlay = <Menu>{overlayMenu}</Menu>;
            }
            body = (
                // @ts-expect-error outdated @types/react
                <Dropdown placement={dropdownPlacement ? dropdownPlacement : "bottomRight"} overlay={overlay}>
                    {body}
                </Dropdown>
            );
        }

        return body;
    }
}
