<?php
namespace DevOwl\CookieConsentManagement\settings;

use DevOwl\CookieConsentManagement\services\Blocker;
use DevOwl\CookieConsentManagement\services\ServiceGroup;

/**
 * Abstract implementation of the settings for general consent management (e.g. is cookie banner active?).
 */
abstract class AbstractGeneral extends BaseSettings {
    const TERRITORIAL_LEGAL_BASIS_GDPR = 'gdpr-eprivacy';
    const TERRITORIAL_LEGAL_BASIS_DSG_SWITZERLAND = 'dsg-switzerland';
    const LEGAL_BASIS_ALLOWED = [self::TERRITORIAL_LEGAL_BASIS_GDPR, self::TERRITORIAL_LEGAL_BASIS_DSG_SWITZERLAND];

    /**
     * Is the banner active?
     *
     * @return boolean
     */
    abstract public function isBannerActive();

    /**
     * Is the content blocker active?
     *
     * @return boolean
     */
    abstract public function isBlockerActive();

    /**
     * Get configured legal basis.
     *
     * @return string[]
     */
    abstract public function getTerritorialLegalBasis();

    /**
     * Get configured operator country.
     *
     * @return string
     */
    abstract public function getOperatorCountry();

    /**
     * Get configured operator contact address.
     *
     * @return string
     */
    abstract public function getOperatorContactAddress();

    /**
     * Get configured operator contact phone.
     *
     * @return string
     */
    abstract public function getOperatorContactPhone();

    /**
     * Get configured operator contact email.
     *
     * @return string
     */
    abstract public function getOperatorContactEmail();

    /**
     * Get the operator contact form page ID.
     *
     * @return int
     */
    abstract public function getOperatorContactFormId();

    /**
     * Get the cookie policy page ID.
     *
     * @return int
     */
    abstract public function getCookiePolicyId();

    /**
     * Get the operator contact form page URL.
     *
     * @param mixed $default
     * @return mixed
     */
    abstract public function getOperatorContactFormUrl($default = false);

    /**
     * Get the cookie policy page URL.
     *
     * @param mixed $default
     * @return mixed
     */
    abstract public function getCookiePolicyUrl($default = false);

    /**
     * Get an array of hidden page ids (not imprint and privacy policy, there are own options!).
     *
     * @return int[]
     */
    abstract public function getAdditionalPageHideIds();

    /**
     * Get the option "Load services after consent via".
     *
     * @return string
     */
    abstract public function getSetCookiesViaManager();

    /**
     * Get the list of service groups and their services.
     *
     * @return ServiceGroup[]
     */
    abstract public function getServiceGroups();

    /**
     * Get the list of content blockers.
     *
     * @return Blocker[]
     */
    abstract public function getBlocker();

    /**
     * Get the list of banner links which should be shown in cookie banner and content blocker.
     *
     * @return BannerLink[]
     */
    abstract public function getBannerLinks();

    /**
     * Get the list of languages in which this website is reachable. This is also used for the language switcher.
     *
     * @return Language[]
     */
    abstract public function getLanguages();

    /**
     * Get the essential service groups.
     */
    public function getEssentialServiceGroup() {
        foreach ($this->getServiceGroups() as $group) {
            if ($group->isEssential()) {
                return $group;
            }
        }

        return null;
    }

    /**
     * Get the list of service groups and their services which are not created by default (custom groups).
     *
     * @return ServiceGroup[]
     */
    public function getNonDefaultServiceGroups() {
        $result = [];

        foreach ($this->getServiceGroups() as $group) {
            if (!$group->isDefault()) {
                $result[] = $group;
            }
        }

        return $result;
    }
}
