<?php
namespace DevOwl\HeadlessContentBlocker;

use DevOwl\FastHtmlTag\finder\match\AbstractMatch;
use DevOwl\FastHtmlTag\finder\match\ScriptInlineMatch;
use DevOwl\FastHtmlTag\finder\match\StyleInlineMatch;
use DevOwl\HeadlessContentBlocker\finder\match\StyleInlineAttributeMatch;
use DevOwl\HeadlessContentBlocker\matcher\AbstractMatcher;
use DevOwl\HeadlessContentBlocker\matcher\ScriptInlineMatcher;
use DevOwl\HeadlessContentBlocker\matcher\StyleInlineAttributeMatcher;
use DevOwl\HeadlessContentBlocker\matcher\StyleInlineMatcher;
use Sabberworm\CSS\CSSList\Document;

/**
 * Describes a plugin for `HeadlessContentBlocker`. You can override each of this methods.
 */
abstract class AbstractPlugin {
    private $headlessContentBlocker;

    /**
     * C'tor.
     *
     * @param HeadlessContentBlocker $headlessContentBlocker
     */
    final public function __construct($headlessContentBlocker) {
        $this->headlessContentBlocker = $headlessContentBlocker;
    }

    /**
     * Before the content blocker gets setup.
     *
     * @codeCoverageIgnore
     */
    public function setup() {
        // Silence is golden.
    }

    /**
     * The content blocker got setup completely.
     *
     * @codeCoverageIgnore
     */
    public function afterSetup() {
        // Silence is golden.
    }

    /**
     * Initialize your plugin add e.g. new visual parent definitions.
     */
    public function init() {
        // Silence is golden.
    }

    /**
     * Allows to modify the HTML after the content blocker has done its job.
     *
     * @param string $html
     * @codeCoverageIgnore
     */
    public function modifyHtmlAfterProcessing($html) {
        return $html;
    }

    /**
     * Called before a match got found.
     *
     * @param AbstractMatcher $matcher
     * @param AbstractMatch $match
     * @codeCoverageIgnore
     */
    public function beforeMatch($matcher, $match) {
        // Silence is golden.
    }

    /**
     * Called after a match got blocked.
     *
     * @param BlockedResult $result
     * @param AbstractMatcher $matcher
     * @param AbstractMatch $match
     * @codeCoverageIgnore
     */
    public function blockedMatch($result, $matcher, $match) {
        // Silence is golden.
    }

    /**
     * Called after a match got not blocked. Opposite of `blockedMatch`.
     *
     * @param BlockedResult $result
     * @param AbstractMatcher $matcher
     * @param AbstractMatch $match
     * @codeCoverageIgnore
     */
    public function notBlockedMatch($result, $matcher, $match) {
        // Silence is golden.
    }

    /**
     * Called after a match got found and the matcher decided, if it should be blocked or not.
     *
     * @param BlockedResult $result
     * @param AbstractMatcher $matcher
     * @param AbstractMatch $match
     * @return BlockedResult
     * @codeCoverageIgnore
     */
    public function checkResult($result, $matcher, $match) {
        return $result;
    }

    /**
     * Keep attributes for a specific match.
     *
     * @param string[] $keepAttributes
     * @param AbstractMatcher $matcher
     * @param AbstractMatch $match
     * @return string[]
     * @codeCoverageIgnore
     */
    public function keepAlwaysAttributes($keepAttributes, $matcher, $match) {
        return $keepAttributes;
    }

    /**
     * Skip inline script by variable name.
     *
     * @param string[] $names
     * @param ScriptInlineMatcher $matcher
     * @param ScriptInlineMatch $match
     * @return string[]
     * @codeCoverageIgnore
     */
    public function skipInlineScriptVariableAssignment($names, $matcher, $match) {
        return $names;
    }

    /**
     * This method is run while iterating blockables and check if rules are found within a string.
     * It allows you to add additional blockables to the result even if `isAllowMultipleBlockerResults` is `false`.
     *
     * Note: All required IDs from each matching Blockable will be merged together, but only the ID of the first Blockable found
     * will be assigned as the `consent-id` of the element.
     *
     * @param AbstractMatcher $matcher
     * @param BlockedResult $result
     * @param string $string
     * @param boolean $useContainsRegularExpression
     * @param boolean $multilineRegexp
     * @param string[] $useRegularExpressionFromMap
     * @param AbstractBlockable[] $useBlockables
     * @param boolean $allowMultiple
     * @codeCoverageIgnore
     */
    public function iterateBlockablesInString(
        $matcher,
        $result,
        $string,
        $useContainsRegularExpression,
        $multilineRegexp,
        $useRegularExpressionFromMap,
        $useBlockables,
        $allowMultiple
    ) {
        // Silence is golden.
    }

    /**
     * Do not extract blocked rules of a CSS inline script to a second document.
     *
     * @param boolean $extract
     * @param StyleInlineMatcher|StyleInlineAttributeMatcher $matcher
     * @param StyleInlineMatch|StyleInlineAttributeMatch $match
     * @return boolean
     * @codeCoverageIgnore
     */
    public function inlineStyleShouldBeExtracted($extract, $matcher, $match) {
        return $extract;
    }

    /**
     * Allows to modify blocked CSS documents.
     *
     * @param Document $document
     * @param Document $extractedDocument
     * @param StyleInlineMatcher|StyleInlineAttributeMatcher $matcher
     * @param StyleInlineMatch|StyleInlineAttributeMatch $match
     * @return boolean
     * @codeCoverageIgnore
     */
    public function inlineStyleModifyDocuments($document, $extractedDocument, $matcher, $match) {
        // Silence is golden.
    }

    /**
     * Decide if a URL in a CSS rule should be blocked.
     *
     * @param BlockedResult $result
     * @param string $url
     * @param StyleInlineMatcher|StyleInlineAttributeMatcher $matcher
     * @param StyleInlineMatch|StyleInlineAttributeMatch $match
     * @return boolean
     * @codeCoverageIgnore
     */
    public function inlineStyleBlockRule($result, $url, $matcher, $match) {
        return $result;
    }

    /**
     * Set a visual parent for a specific match.
     *
     * @param boolean|string|number $visualParent
     * @param AbstractMatcher $matcher
     * @param AbstractMatch $match
     * @return boolean|string|number
     * @codeCoverageIgnore
     */
    public function visualParent($visualParent, $matcher, $match) {
        return $visualParent;
    }

    /**
     * Allows to modify the passed string expression to a blockable.
     *
     * @param string $expression
     * @param AbstractBlockable $blockable
     * @return string
     * @codeCoverageIgnore
     */
    public function blockableStringExpression($expression, $blockable) {
        return $expression;
    }

    /**
     * Allows to add an additional check before a blockable and expression gets added to a `BlockedResult`.
     *
     * @param BlockedResult $result
     * @param AbstractBlockable $blockable
     * @param string $expression
     * @param AbstractMatcher $matcher
     * @return boolean
     * @codeCoverageIgnore
     */
    public function beforeSetBlockedInResult($result, $blockable, $expression, $matcher) {
        return true;
    }

    /**
     * Modify the blockables array before it gets registered.
     *
     * @param AbstractBlockable[] $blockables
     * @return AbstractBlockable[]
     * @codeCoverageIgnore
     */
    public function modifyBlockables($blockables) {
        return $blockables;
    }

    /**
     * Getter.
     */
    final public function getHeadlessContentBlocker() {
        return $this->headlessContentBlocker;
    }
}
