import { useMemo } from "react";

import { A11yNotice } from "./notice.js";
import { useA11yDispatcher } from "../../hooks/useA11yDispatcher.js";
import { useCustomizeValues } from "../../hooks/useCustomizeValues.js";
import { getSidebarCustomize } from "../../sidebar/getSidebarCustomize.js";
import { __ } from "../../utils/i18n.js";

import type { ComponentProps, FC } from "react";

const A11yFontSize: FC<{
    /**
     * List of setting IDs. The first one needs to be the settings which expose a
     * font size in `px`. All other settings are useful for `modifier`.
     *
     * This property does not listen to changes and is used one-time.
     */
    settings: string[];
    minimumPx: number;
    severity: ComponentProps<typeof A11yNotice>["severity"];
    /**
     * Allows you to modify the value for the settings at runtime.
     */
    modifier?: (fontSize: number, values: any[], getSetting: <T>(setting: string) => T) => number;
}> = ({ settings, severity, minimumPx, modifier }) => {
    const customize = getSidebarCustomize();
    const values = useCustomizeValues(settings);

    const hasError = useMemo(() => {
        const fontSize = modifier
            ? modifier(
                  values[settings[0]].value,
                  settings.slice(1).reduce((p, c) => {
                      p.push(values[c].value);
                      return p;
                  }, [] as any[]),
                  (setting) => customize(setting).get(),
              )
            : (values[settings[0]].value as number);
        return fontSize < minimumPx ? minimumPx : false;
    }, [JSON.stringify(values), minimumPx]);

    useA11yDispatcher({
        setting: settings[0],
        valid: hasError === false,
    });

    if (hasError === false) {
        return null;
    }

    return (
        <A11yNotice
            severity={severity}
            text={__(
                'We recommend a font size of at least %d px to comply with the {{aPerc}}"perceptibility" of the WCAG 2.2 standard{{/aPerc}} and thereby meet the {{aEuMinimum}}EU\'s minimum legal requirements for accessibility{{/aEuMinimum}}.',
                hasError,
            )}
        />
    );
};

export { A11yFontSize };
