import { Divider, Spin } from "antd";
import { observer } from "mobx-react";
import { useEffect } from "react";

import { PluginUpdateForm } from "./form/form.js";
import { PluginUpdateLicenseList } from "./licenseList.js";
import { useStores } from "../../store/stores.js";
import { __, _i } from "../../utils/i18n.js";

import type { ComponentPropsWithRef, FC } from "react";

const PluginUpdateErrorNotice: FC<{ slug: string }> = observer(({ slug }) => {
    const error = useStores().pluginUpdateStore.pluginUpdateFetchErrors.get(slug);
    const errorMessage = error?.responseJSON?.message;

    return (
        <div className="notice notice-error inline below-h2 notice-alt" style={{ margin: "20px 0" }}>
            <p>
                {_i(
                    __(
                        "Something has gone wrong%s. It could be that you have configured your WordPress instance in such a way that you have blocked the WordPress REST API. Find out how to deal with this {{a}}here{{/a}}.",
                        errorMessage ? ` (${errorMessage})` : "",
                    ),
                    {
                        a: (
                            <a
                                href={__("https://devowl.io/knowledge-base/wordpress-rest-api-does-not-respond/")}
                                target="_blank"
                                rel="noreferrer"
                            ></a>
                        ),
                    },
                )}
            </p>
        </div>
    );
});

/**
 * Embed the license form and list of activated licenses into your app.
 * Please consider, that this embed does not provide any buttons, please provide
 * them yourself with `<button form="license-form-$slug"`>
 */
const PluginUpdateEmbed: FC<{
    formProps?: Omit<ComponentPropsWithRef<typeof PluginUpdateForm>, "pluginUpdate">;
    listProps?: Omit<ComponentPropsWithRef<typeof PluginUpdateLicenseList>, "pluginUpdate">;
    slug: string;
}> = observer(({ formProps = {}, listProps = {}, slug }) => {
    const { pluginUpdateStore } = useStores();
    const { busy, pluginUpdates, pluginUpdateFetchErrors } = pluginUpdateStore;
    const pluginUpdate = pluginUpdates.get(slug);
    const showHeadline = pluginUpdate?.unlicensedEntries.length > 0 && pluginUpdate?.licensedEntries.length > 0;

    useEffect(() => {
        try {
            pluginUpdateStore.fetchPluginUpdate(slug);
        } catch (e) {
            // Silence is golden.
        }
    }, [slug]);

    return pluginUpdateFetchErrors.get(slug) ? (
        <PluginUpdateErrorNotice slug={slug} />
    ) : busy || !pluginUpdate ? (
        <Spin spinning />
    ) : (
        <div>
            {showHeadline && (
                <Divider type="horizontal" orientation="left" style={{ marginTop: 0 }}>
                    {__("Not yet licensed")}
                </Divider>
            )}
            <PluginUpdateForm {...formProps} pluginUpdate={pluginUpdate} />
            {showHeadline && (
                <Divider type="horizontal" orientation="left">
                    {__("Already licensed")}
                </Divider>
            )}
            {pluginUpdate.licensedEntries.length > 0 && (
                <PluginUpdateLicenseList {...listProps} pluginUpdate={pluginUpdate} />
            )}
        </div>
    );
});

export { PluginUpdateErrorNotice, PluginUpdateEmbed };
